<?php

namespace App\Exports;

use App\Models\AuditLog;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class AuditLogExport implements FromCollection, WithHeadings, WithMapping, WithStyles, WithTitle, ShouldAutoSize
{
    protected $auditLogs;
    protected $title;

    public function __construct($auditLogs, $title = 'تقرير سجل المراجعة')
    {
        $this->auditLogs = $auditLogs;
        $this->title = $title;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return $this->auditLogs;
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'الرقم',
            'نوع الحدث',
            'وصف الحدث',
            'نوع النموذج',
            'وصف النموذج',
            'معرف النموذج',
            'المستخدم',
            'عنوان IP',
            'المتصفح',
            'الرابط',
            'الوصف',
            'العلامات',
            'تاريخ الحدث',
        ];
    }

    /**
     * @param mixed $auditLog
     * @return array
     */
    public function map($auditLog): array
    {
        return [
            $auditLog->id,
            $auditLog->event,
            $auditLog->event_description,
            $auditLog->auditable_type,
            $auditLog->model_type_description,
            $auditLog->auditable_id,
            $auditLog->user->name ?? 'غير محدد',
            $auditLog->ip_address,
            $this->getBrowserName($auditLog->user_agent),
            $auditLog->url,
            $auditLog->description,
            is_array($auditLog->tags) ? implode(', ', $auditLog->tags) : '',
            $auditLog->created_at->format('Y-m-d H:i:s'),
        ];
    }

    /**
     * @param Worksheet $sheet
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        return [
            // تنسيق العنوان
            1 => [
                'font' => [
                    'bold' => true,
                    'size' => 12,
                    'color' => ['rgb' => 'FFFFFF'],
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '8E44AD'],
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
            // تنسيق الخلايا
            'A:M' => [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
            // تنسيق التاريخ
            'M:M' => [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_RIGHT,
                ],
            ],
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return $this->title;
    }

    /**
     * Get simplified browser name from user agent
     */
    private function getBrowserName($userAgent)
    {
        if (empty($userAgent)) {
            return 'غير محدد';
        }

        if (strpos($userAgent, 'Chrome') !== false) {
            return 'Chrome';
        } elseif (strpos($userAgent, 'Firefox') !== false) {
            return 'Firefox';
        } elseif (strpos($userAgent, 'Safari') !== false) {
            return 'Safari';
        } elseif (strpos($userAgent, 'Edge') !== false) {
            return 'Edge';
        } elseif (strpos($userAgent, 'Opera') !== false) {
            return 'Opera';
        } else {
            return 'أخرى';
        }
    }
}