<?php

namespace App\Exceptions;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class InvoiceCashFlowException extends Exception
{
    protected $errorCode;
    protected $context = [];

    public function __construct($message = "", $errorCode = 'CASH_FLOW_ERROR', array $context = [], $code = 0, \Throwable $previous = null)
    {
        parent::__construct($message, $code, $previous);
        $this->errorCode = $errorCode;
        $this->context = $context;
    }

    /**
     * إنشاء استثناء لخطأ في ربط الدفعة
     */
    public static function paymentLinkingFailed($message, array $context = []): self
    {
        return new self($message, 'PAYMENT_LINKING_FAILED', $context);
    }

    /**
     * إنشاء استثناء لخطأ في التحقق من صحة البيانات
     */
    public static function validationFailed($message, array $context = []): self
    {
        return new self($message, 'VALIDATION_FAILED', $context);
    }

    /**
     * إنشاء استثناء لخطأ في تحليل التدفق النقدي
     */
    public static function analysissFailed($message, array $context = []): self
    {
        return new self($message, 'CASH_FLOW_ANALYSIS_FAILED', $context);
    }

    /**
     * إنشاء استثناء لخطأ في تصدير التقرير
     */
    public static function exportFailed($message, array $context = []): self
    {
        return new self($message, 'EXPORT_FAILED', $context);
    }

    /**
     * إنشاء استثناء لخطأ في البحث المتقدم
     */
    public static function searchFailed($message, array $context = []): self
    {
        return new self($message, 'ADVANCED_SEARCH_FAILED', $context);
    }

    /**
     * معالجة الاستثناء للاستجابة HTTP
     */
    public function render(Request $request): Response
    {
        // تسجيل الخطأ
        \Log::error("Invoice Cash Flow Error: {$this->errorCode}", [
            'message' => $this->getMessage(),
            'context' => $this->context,
            'user_id' => auth()->id(),
            'request_url' => $request->url(),
            'request_method' => $request->method(),
        ]);

        // إذا كان طلب AJAX، إرجاع JSON
        if ($request->expectsJson()) {
            return response()->json([
                'error' => true,
                'message' => $this->getMessage(),
                'error_code' => $this->errorCode,
                'context' => $this->context
            ], 400);
        }

        // إرجاع المستخدم للصفحة السابقة مع رسالة خطأ
        return back()
            ->withErrors(['error' => $this->getMessage()])
            ->withInput();
    }

    /**
     * الحصول على كود الخطأ
     */
    public function getErrorCode(): string
    {
        return $this->errorCode;
    }

    /**
     * الحصول على السياق
     */
    public function getContext(): array
    {
        return $this->context;
    }

    /**
     * إضافة معلومات إضافية للسياق
     */
    public function addContext(string $key, $value): self
    {
        $this->context[$key] = $value;
        return $this;
    }

    /**
     * تحويل الاستثناء إلى مصفوفة
     */
    public function toArray(): array
    {
        return [
            'message' => $this->getMessage(),
            'error_code' => $this->errorCode,
            'context' => $this->context,
            'file' => $this->getFile(),
            'line' => $this->getLine(),
        ];
    }
}