<?php

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * Define the application's command schedule.
     */
    protected function schedule(Schedule $schedule): void
    {
        // النظام المحاسبي - المهام المجدولة
        
        // توليد التقارير المجدولة كل ساعة
        $schedule->command('accounting:generate-reports')
                 ->hourly()
                 ->withoutOverlapping()
                 ->runInBackground()
                 ->emailOutputOnFailure(config('accounting.admin_email'))
                 ->description('توليد التقارير المحاسبية المجدولة');

        // صيانة النظام المحاسبي يومياً في الساعة 2:00 صباحاً
        $schedule->command('accounting:maintenance all --force')
                 ->dailyAt('02:00')
                 ->withoutOverlapping()
                 ->runInBackground()
                 ->emailOutputOnFailure(config('accounting.admin_email'))
                 ->description('صيانة دورية للنظام المحاسبي');

        // تحديث أرصدة الحسابات كل 6 ساعات
        $schedule->command('accounting:maintenance balances --force')
                 ->everySixHours()
                 ->withoutOverlapping()
                 ->runInBackground()
                 ->description('تحديث أرصدة الحسابات');

        // تحديث الميزانيات كل 4 ساعات خلال ساعات العمل
        $schedule->command('accounting:maintenance budgets --force')
                 ->cron('0 */4 8-18 * * 1-5') // كل 4 ساعات من 8 صباحاً إلى 6 مساءً، أيام العمل
                 ->withoutOverlapping()
                 ->runInBackground()
                 ->description('تحديث الميزانيات خلال ساعات العمل');

        // تنظيف سجل المراجعة أسبوعياً
        $schedule->command('accounting:maintenance audit-logs --force')
                 ->weekly()
                 ->sundays()
                 ->at('03:00')
                 ->withoutOverlapping()
                 ->runInBackground()
                 ->description('تنظيف سجل المراجعة القديم');

        // تنظيف ملفات التقارير القديمة أسبوعياً
        $schedule->command('accounting:maintenance reports --force')
                 ->weekly()
                 ->sundays()
                 ->at('04:00')
                 ->withoutOverlapping()
                 ->runInBackground()
                 ->description('تنظيف ملفات التقارير القديمة');

        // إنشاء نسخة احتياطية يومياً
        $schedule->command('accounting:maintenance backup --force')
                 ->dailyAt('01:00')
                 ->withoutOverlapping()
                 ->runInBackground()
                 ->emailOutputOnFailure(config('accounting.admin_email'))
                 ->description('إنشاء نسخة احتياطية للنظام المحاسبي');

        // إرسال تقارير الميزانيات التي تجاوزت الحد الحرج
        $schedule->call(function () {
            $criticalBudgets = \App\Models\Budget::overCriticalThreshold()
                ->with(['costCenter', 'account', 'creator'])
                ->get();

            if ($criticalBudgets->isNotEmpty()) {
                // إرسال تنبيهات للإدارة
                $adminEmails = config('accounting.budget_alert_emails', []);
                
                foreach ($adminEmails as $email) {
                    // Mail::to($email)->send(new CriticalBudgetAlert($criticalBudgets));
                }

                // تسجيل في سجل المراجعة
                \App\Models\AuditLog::logCustomEvent(
                    'critical_budget_alert',
                    "تم العثور على {$criticalBudgets->count()} ميزانية تجاوزت الحد الحرج",
                    ['budget_count' => $criticalBudgets->count()],
                    ['budget_alerts', 'automation']
                );
            }
        })->dailyAt('09:00')
          ->name('critical-budget-alerts')
          ->description('فحص الميزانيات الحرجة وإرسال التنبيهات');

        // فحص السنوات المالية المنتهية وإنشاء سنوات جديدة
        $schedule->call(function () {
            $endingSoon = \App\Models\FiscalYear::where('is_current', true)
                ->where('end_date', '<=', now()->addDays(30))
                ->where('end_date', '>=', now())
                ->get();

            foreach ($endingSoon as $fiscalYear) {
                // التحقق من وجود سنة مالية تالية
                $nextYear = \App\Models\FiscalYear::where('start_date', '>', $fiscalYear->end_date)
                    ->orderBy('start_date')
                    ->first();

                if (!$nextYear) {
                    // إنشاء السنة المالية التالية
                    $fiscalYear->createNext();
                    
                    // تسجيل في سجل المراجعة
                    \App\Models\AuditLog::logCustomEvent(
                        'fiscal_year_auto_created',
                        "تم إنشاء سنة مالية جديدة تلقائياً بعد {$fiscalYear->name}",
                        ['previous_fiscal_year_id' => $fiscalYear->id],
                        ['fiscal_year', 'automation']
                    );
                }
            }
        })->monthly()
          ->name('fiscal-year-management')
          ->description('إدارة السنوات المالية التلقائية');

        // تحديث حالة الميزانيات المنتهية
        $schedule->call(function () {
            $expiredBudgets = \App\Models\Budget::where('is_active', true)
                ->where('end_date', '<', now())
                ->get();

            foreach ($expiredBudgets as $budget) {
                $budget->update(['is_active' => false]);
                
                \App\Models\AuditLog::logEvent(
                    'deactivated',
                    $budget,
                    'تم إلغاء تفعيل الميزانية تلقائياً بسبب انتهاء الفترة'
                );
            }

            if ($expiredBudgets->isNotEmpty()) {
                \App\Models\AuditLog::logCustomEvent(
                    'budgets_auto_deactivated',
                    "تم إلغاء تفعيل {$expiredBudgets->count()} ميزانية منتهية تلقائياً",
                    ['deactivated_count' => $expiredBudgets->count()],
                    ['budget_management', 'automation']
                );
            }
        })->dailyAt('00:30')
          ->name('budget-expiry-check')
          ->description('فحص الميزانيات المنتهية وإلغاء تفعيلها');

        // مراقبة أداء النظام المحاسبي
        $schedule->call(function () {
            $stats = [
                'accounts_count' => \App\Models\Account::count(),
                'active_accounts' => \App\Models\Account::where('is_active', true)->count(),
                'journal_entries_today' => \App\Models\JournalEntry::whereDate('created_at', today())->count(),
                'pending_approvals' => \App\Models\JournalEntry::where('is_approved', false)->count(),
                'active_budgets' => \App\Models\Budget::where('is_active', true)->count(),
                'critical_budgets' => \App\Models\Budget::overCriticalThreshold()->count(),
                'audit_logs_today' => \App\Models\AuditLog::whereDate('created_at', today())->count(),
                'scheduled_reports' => \App\Models\SavedReport::scheduled()->active()->count(),
            ];

            // تسجيل إحصائيات الأداء
            \App\Models\AuditLog::logCustomEvent(
                'daily_performance_stats',
                'إحصائيات أداء النظام المحاسبي اليومية',
                $stats,
                ['performance', 'monitoring']
            );

            // فحص التحذيرات
            $warnings = [];
            
            if ($stats['pending_approvals'] > 50) {
                $warnings[] = "عدد كبير من القيود في انتظار الاعتماد: {$stats['pending_approvals']}";
            }
            
            if ($stats['critical_budgets'] > 0) {
                $warnings[] = "ميزانيات تجاوزت الحد الحرج: {$stats['critical_budgets']}";
            }

            if (!empty($warnings)) {
                \App\Models\AuditLog::logCustomEvent(
                    'system_warnings',
                    'تحذيرات النظام المحاسبي',
                    ['warnings' => $warnings],
                    ['warnings', 'monitoring']
                );
            }
        })->dailyAt('23:55')
          ->name('accounting-performance-monitoring')
          ->description('مراقبة أداء النظام المحاسبي');

        // مهام Laravel الافتراضية
        
        // $schedule->command('inspire')->hourly();
        
        // تنظيف ملفات التخزين المؤقت
        $schedule->command('cache:clear')
                 ->weekly()
                 ->sundays()
                 ->at('05:00')
                 ->description('تنظيف ذاكرة التخزين المؤقت');

        // تحسين قاعدة البيانات
        $schedule->command('optimize:clear')
                 ->weekly()
                 ->sundays()
                 ->at('05:30')
                 ->description('تحسين أداء التطبيق');
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}