<?php

namespace App\Console\Commands;

use App\Services\ZatcaService;
use App\Models\ZatcaSettings;
use Illuminate\Console\Command;
use Exception;

class ZatcaTestCommand extends Command
{
    /**
     * اسم الأمر والتوقيع
     */
    protected $signature = 'zatca:test {--connection : اختبار الاتصال فقط} {--signing : اختبار التوقيع فقط} {--all : اختبار شامل}';

    /**
     * وصف الأمر
     */
    protected $description = 'اختبار وتشخيص نظام ZATCA للفوترة الإلكترونية';

    /**
     * تنفيذ الأمر
     */
    public function handle()
    {
        $this->info('🔧 بدء اختبار نظام ZATCA...');
        $this->newLine();

        // فحص الخيارات
        $testConnection = $this->option('connection');
        $testSigning = $this->option('signing');
        $testAll = $this->option('all');

        if (!$testConnection && !$testSigning && !$testAll) {
            $testAll = true; // تشغيل جميع الاختبارات افتراضياً
        }

        $allPassed = true;

        try {
            // 1. فحص متطلبات النظام
            if ($testAll) {
                $this->info('📋 فحص متطلبات النظام...');
                $allPassed &= $this->checkSystemRequirements();
                $this->newLine();
            }

            // 2. فحص إعدادات ZATCA
            if ($testAll || $testConnection) {
                $this->info('⚙️  فحص إعدادات ZATCA...');
                $settings = $this->checkZatcaSettings();
                if (!$settings) {
                    $allPassed = false;
                } else {
                    $this->newLine();
                }
            }

            // 3. اختبار الاتصال
            if (($testAll || $testConnection) && isset($settings)) {
                $this->info('🌐 اختبار الاتصال مع ZATCA...');
                $allPassed &= $this->testZatcaConnection($settings);
                $this->newLine();
            }

            // 4. اختبار أداة Fatoora
            if ($testAll || $testSigning) {
                $this->info('📝 اختبار أداة التوقيع Fatoora...');
                $allPassed &= $this->testFatooraTools();
                $this->newLine();
            }

            // 5. اختبار إنشاء XML
            if ($testAll || $testSigning) {
                $this->info('📄 اختبار إنشاء XML...');
                $allPassed &= $this->testXmlGeneration();
                $this->newLine();
            }

            // النتيجة النهائية
            if ($allPassed) {
                $this->info('✅ جميع الاختبارات نجحت! نظام ZATCA جاهز للاستخدام.');
            } else {
                $this->error('❌ بعض الاختبارات فشلت. يرجى مراجعة الأخطاء أعلاه.');
                return Command::FAILURE;
            }

        } catch (Exception $e) {
            $this->error('💥 خطأ عام: ' . $e->getMessage());
            return Command::FAILURE;
        }

        return Command::SUCCESS;
    }

    /**
     * فحص متطلبات النظام
     */
    private function checkSystemRequirements(): bool
    {
        $requirements = [
            'PHP Version >= 8.1' => version_compare(PHP_VERSION, '8.1.0', '>='),
            'OpenSSL Extension' => extension_loaded('openssl'),
            'DOM Extension' => extension_loaded('dom'),
            'MBString Extension' => extension_loaded('mbstring'),
            'cURL Extension' => extension_loaded('curl'),
            'JSON Extension' => extension_loaded('json'),
        ];

        $allGood = true;

        foreach ($requirements as $requirement => $status) {
            if ($status) {
                $this->line("  ✅ {$requirement}");
            } else {
                $this->line("  ❌ {$requirement}");
                $allGood = false;
            }
        }

        // فحص مجلدات التخزين
        $directories = [
            'storage/app/invoices' => storage_path('app/invoices'),
            'storage/logs' => storage_path('logs'),
        ];

        foreach ($directories as $name => $path) {
            if (is_dir($path) && is_writable($path)) {
                $this->line("  ✅ {$name} (قابل للكتابة)");
            } else {
                $this->line("  ❌ {$name} (غير موجود أو غير قابل للكتابة)");
                $allGood = false;
            }
        }

        return $allGood;
    }

    /**
     * فحص إعدادات ZATCA
     */
    private function checkZatcaSettings(): ?ZatcaSettings
    {
        try {
            $settings = ZatcaSettings::first();
            
            if (!$settings) {
                $this->error('  ❌ لم يتم العثور على إعدادات ZATCA في قاعدة البيانات');
                $this->line('     قم بإنشاء إعدادات ZATCA من لوحة التحكم أولاً');
                return null;
            }

            $this->line("  ✅ تم العثور على إعدادات ZATCA");
            $this->line("  📋 البيئة: {$settings->environment}");
            $this->line("  🏢 معرف الشركة: {$settings->company_id}");
            
            // فحص الحقول المطلوبة
            $requiredFields = [
                'environment' => $settings->environment,
                'company_id' => $settings->company_id,
            ];

            if ($settings->environment === 'sandbox') {
                $requiredFields['username'] = $settings->username;
                $requiredFields['password'] = $settings->password;
            } else {
                $requiredFields['client_id'] = $settings->client_id;
                $requiredFields['client_secret'] = $settings->client_secret;
            }

            $missingFields = [];
            foreach ($requiredFields as $field => $value) {
                if (empty($value)) {
                    $missingFields[] = $field;
                }
            }

            if (!empty($missingFields)) {
                $this->warn("  ⚠️  حقول ناقصة: " . implode(', ', $missingFields));
            }

            return $settings;

        } catch (Exception $e) {
            $this->error("  ❌ خطأ في فحص الإعدادات: " . $e->getMessage());
            return null;
        }
    }

    /**
     * اختبار الاتصال مع ZATCA
     */
    private function testZatcaConnection(ZatcaSettings $settings): bool
    {
        try {
            $zatcaService = new ZatcaService($settings);
            
            // فحص بيانات الاعتماد أولاً
            if ($settings->environment === 'sandbox') {
                $this->line("  🔐 فحص بيانات الاعتماد للبيئة التجريبية...");
                $credentialsResult = $zatcaService->setupSandboxCredentials();
                
                if ($credentialsResult['success']) {
                    $this->line("  ✅ بيانات الاعتماد صحيحة");
                } else {
                    $this->error("  ❌ بيانات الاعتماد غير صحيحة: " . $credentialsResult['message']);
                    
                    if (isset($credentialsResult['data']['suggestion'])) {
                        $this->line("  💡 اقتراح: " . $credentialsResult['data']['suggestion']);
                    }
                    
                    return false;
                }
            }
            
            // اختبار الاتصال العام
            $result = $zatcaService->testConnection();

            if ($result['success']) {
                $this->line("  ✅ الاتصال ناجح: " . $result['message']);
                return true;
            } else {
                $this->error("  ❌ فشل الاتصال: " . $result['message']);
                return false;
            }

        } catch (Exception $e) {
            $this->error("  ❌ خطأ في الاتصال: " . $e->getMessage());
            return false;
        }
    }

    /**
     * اختبار أداة Fatoora
     */
    private function testFatooraTools(): bool
    {
        // فحص وجود أداة fatoora
        $fatooraPaths = [
            'C:\\fatoora\\fatoora.exe',
            'C:\\Program Files\\fatoora\\fatoora.exe',
            '/usr/local/bin/fatoora',
            '/usr/bin/fatoora',
            storage_path('app/fatoora/fatoora'),
        ];

        $fatooraFound = false;
        foreach ($fatooraPaths as $path) {
            if (file_exists($path)) {
                $this->line("  ✅ تم العثور على أداة Fatoora: {$path}");
                $fatooraFound = true;
                break;
            }
        }

        // فحص PATH
        if (!$fatooraFound) {
            exec('where fatoora 2>nul || which fatoora 2>/dev/null', $output, $returnVar);
            if ($returnVar === 0 && !empty($output)) {
                $this->line("  ✅ أداة Fatoora متوفرة في PATH: " . trim($output[0]));
                $fatooraFound = true;
            }
        }

        if (!$fatooraFound) {
            $this->warn("  ⚠️  لم يتم العثور على أداة Fatoora");
            $this->line("     سيتم استخدام الطريقة البديلة للتوقيع");
            $this->line("     لتحميل أداة Fatoora: راجع FATOORA-SETUP-GUIDE.md");
        }

        // اختبار الطريقة البديلة
        try {
            // إنشاء ملف XML اختبار
            $testXml = '<?xml version="1.0" encoding="UTF-8"?><test>اختبار</test>';
            $testDir = storage_path('app/test');
            
            if (!is_dir($testDir)) {
                mkdir($testDir, 0755, true);
            }

            $inputPath = $testDir . '/test_input.xml';
            $outputPath = $testDir . '/test_output.xml';

            file_put_contents($inputPath, $testXml);

            // محاولة التوقيع البديل
            $settings = ZatcaSettings::first();
            if ($settings) {
                $zatcaService = new ZatcaService($settings);
                
                // استخدام reflection للوصول للطريقة الخاصة
                $reflection = new \ReflectionClass($zatcaService);
                $method = $reflection->getMethod('signInvoiceAlternative');
                $method->setAccessible(true);
                $method->invoke($zatcaService, $inputPath, $outputPath);

                if (file_exists($outputPath)) {
                    $this->line("  ✅ الطريقة البديلة للتوقيع تعمل بنجاح");
                    
                    // تنظيف الملفات التجريبية
                    unlink($inputPath);
                    unlink($outputPath);
                    rmdir($testDir);
                    
                    return true;
                } else {
                    $this->error("  ❌ فشل في إنشاء ملف الإخراج");
                    return false;
                }
            } else {
                $this->warn("  ⚠️  لا توجد إعدادات ZATCA لاختبار التوقيع");
                return false;
            }

        } catch (Exception $e) {
            $this->error("  ❌ خطأ في اختبار التوقيع: " . $e->getMessage());
            return false;
        }
    }

    /**
     * اختبار إنشاء XML
     */
    private function testXmlGeneration(): bool
    {
        try {
            // إنشاء فاتورة تجريبية
            $testInvoice = new \stdClass();
            $testInvoice->invoice_number = 'TEST-001';
            $testInvoice->uuid = '123e4567-e89b-12d3-a456-426614174000';
            $testInvoice->issue_date = now();
            $testInvoice->type = 'simplified';
            $testInvoice->currency = 'SAR';
            $testInvoice->subtotal = 100.00;
            $testInvoice->tax_amount = 15.00;
            $testInvoice->total_amount = 115.00;

            // محاولة إنشاء XML بسيط
            $xml = new \DOMDocument('1.0', 'UTF-8');
            $xml->formatOutput = true;

            $root = $xml->createElement('TestInvoice');
            $root->appendChild($xml->createElement('ID', $testInvoice->invoice_number));
            $root->appendChild($xml->createElement('UUID', $testInvoice->uuid));
            $root->appendChild($xml->createElement('IssueDate', $testInvoice->issue_date->format('Y-m-d')));
            $xml->appendChild($root);

            $xmlString = $xml->saveXML();

            if (!empty($xmlString) && strpos($xmlString, 'TestInvoice') !== false) {
                $this->line("  ✅ إنشاء XML يعمل بنجاح");
                $this->line("  📊 حجم XML: " . strlen($xmlString) . " بايت");
                return true;
            } else {
                $this->error("  ❌ فشل في إنشاء XML صالح");
                return false;
            }

        } catch (Exception $e) {
            $this->error("  ❌ خطأ في إنشاء XML: " . $e->getMessage());
            return false;
        }
    }
}