<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\ZatcaSettings;
use App\Models\DeviceInitializationLog;
use App\Models\Company;

class ZatcaStats extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'zatca:stats {--format=table : Output format (table|json)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'عرض إحصائيات شاملة لنظام ZATCA';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $format = $this->option('format');

        $stats = $this->collectStats();

        if ($format === 'json') {
            $this->line(json_encode($stats, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
            return Command::SUCCESS;
        }

        $this->displayTableStats($stats);
        return Command::SUCCESS;
    }

    /**
     * جمع الإحصائيات
     */
    protected function collectStats(): array
    {
        return [
            'companies' => $this->getCompanyStats(),
            'zatca_settings' => $this->getZatcaSettingsStats(),
            'device_initialization' => $this->getDeviceInitializationStats(),
            'certificates' => $this->getCertificateStats(),
            'system' => $this->getSystemStats(),
        ];
    }

    /**
     * إحصائيات الشركات
     */
    protected function getCompanyStats(): array
    {
        return [
            'total' => Company::count(),
            'active' => Company::where('status', 'active')->count(),
            'with_zatca' => Company::whereHas('zatcaSettings')->count(),
            'without_zatca' => Company::whereDoesntHave('zatcaSettings')->count(),
        ];
    }

    /**
     * إحصائيات إعدادات ZATCA
     */
    protected function getZatcaSettingsStats(): array
    {
        return [
            'total' => ZatcaSettings::count(),
            'active' => ZatcaSettings::where('is_active', true)->count(),
            'connected' => ZatcaSettings::where('connection_status', 'connected')->count(),
            'failed' => ZatcaSettings::where('connection_status', 'failed')->count(),
            'sandbox' => ZatcaSettings::where('environment', 'sandbox')->count(),
            'production' => ZatcaSettings::where('environment', 'production')->count(),
        ];
    }

    /**
     * إحصائيات تهيئة الأجهزة
     */
    protected function getDeviceInitializationStats(): array
    {
        return [
            'total_attempts' => DeviceInitializationLog::count(),
            'verified' => DeviceInitializationLog::where('verification_status', 'verified')->count(),
            'pending' => DeviceInitializationLog::where('verification_status', 'pending')->count(),
            'failed' => DeviceInitializationLog::where('verification_status', 'failed')->count(),
            'expired' => DeviceInitializationLog::where('verification_status', 'expired')->count(),
            'today' => DeviceInitializationLog::whereDate('created_at', today())->count(),
            'this_week' => DeviceInitializationLog::whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
        ];
    }

    /**
     * إحصائيات الشهادات
     */
    protected function getCertificateStats(): array
    {
        return [
            'total' => ZatcaSettings::whereNotNull('certificate_content')->count(),
            'expired' => ZatcaSettings::where('certificate_expires_at', '<', now())->count(),
            'expiring_soon' => ZatcaSettings::whereBetween('certificate_expires_at', [now(), now()->addDays(30)])->count(),
            'valid' => ZatcaSettings::where('certificate_expires_at', '>', now()->addDays(30))->count(),
        ];
    }

    /**
     * إحصائيات النظام
     */
    protected function getSystemStats(): array
    {
        return [
            'zatca_enabled' => config('zatca.enabled', true),
            'default_environment' => config('zatca.default_environment'),
            'logging_enabled' => config('zatca.logging.enabled', true),
            'rate_limit_per_hour' => config('zatca.device_initialization.rate_limit_per_hour', 10),
            'otp_expiry_minutes' => config('zatca.device_initialization.otp_expiry_minutes', 5),
        ];
    }

    /**
     * عرض الإحصائيات في جدول
     */
    protected function displayTableStats(array $stats): void
    {
        $this->info('📊 إحصائيات نظام ZATCA');
        $this->newLine();

        // إحصائيات الشركات
        $this->info('🏢 الشركات:');
        $this->table(
            ['المؤشر', 'العدد'],
            [
                ['إجمالي الشركات', $stats['companies']['total']],
                ['الشركات النشطة', $stats['companies']['active']],
                ['مع إعدادات ZATCA', $stats['companies']['with_zatca']],
                ['بدون إعدادات ZATCA', $stats['companies']['without_zatca']],
            ]
        );

        // إحصائيات إعدادات ZATCA
        $this->info('⚙️ إعدادات ZATCA:');
        $this->table(
            ['المؤشر', 'العدد'],
            [
                ['إجمالي الإعدادات', $stats['zatca_settings']['total']],
                ['نشطة', $stats['zatca_settings']['active']],
                ['متصلة', $stats['zatca_settings']['connected']],
                ['فاشلة', $stats['zatca_settings']['failed']],
                ['بيئة تجريبية', $stats['zatca_settings']['sandbox']],
                ['بيئة إنتاج', $stats['zatca_settings']['production']],
            ]
        );

        // إحصائيات تهيئة الأجهزة
        $this->info('🔧 تهيئة الأجهزة:');
        $this->table(
            ['المؤشر', 'العدد'],
            [
                ['إجمالي المحاولات', $stats['device_initialization']['total_attempts']],
                ['تم التحقق', $stats['device_initialization']['verified']],
                ['معلقة', $stats['device_initialization']['pending']],
                ['فاشلة', $stats['device_initialization']['failed']],
                ['منتهية الصلاحية', $stats['device_initialization']['expired']],
                ['اليوم', $stats['device_initialization']['today']],
                ['هذا الأسبوع', $stats['device_initialization']['this_week']],
            ]
        );

        // إحصائيات الشهادات
        $this->info('📜 الشهادات:');
        $this->table(
            ['المؤشر', 'العدد'],
            [
                ['إجمالي الشهادات', $stats['certificates']['total']],
                ['منتهية الصلاحية', $stats['certificates']['expired']],
                ['تنتهي قريباً (30 يوم)', $stats['certificates']['expiring_soon']],
                ['صالحة', $stats['certificates']['valid']],
            ]
        );
    }
}