<?php

namespace App\Console\Commands;

use App\Models\Invoice;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class UpdateQrCodesForPhase2 extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'zatca:update-qr-codes 
                           {--dry-run : عرض النتائج بدون تطبيق التغييرات}
                           {--force : إجبار التحديث لجميع الفواتير}
                           {--invoice= : تحديث فاتورة محددة بالرقم}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'تحديث QR Codes للفواتير للتوافق مع ZATCA Phase 2';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $dryRun = $this->option('dry-run');
        $force = $this->option('force');
        $invoiceId = $this->option('invoice');

        $this->info('🔄 بدء تحديث QR Codes للتوافق مع ZATCA Phase 2...');
        
        if ($dryRun) {
            $this->warn('⚠️  وضع المعاينة - لن يتم تطبيق أي تغييرات');
        }

        // تحديد الفواتير المراد تحديثها
        $query = Invoice::with(['company', 'customer']);
        
        if ($invoiceId) {
            $query->where('id', $invoiceId);
        } elseif (!$force) {
            // تحديث الفواتير المرسلة لـ ZATCA فقط إذا لم يتم استخدام force
            $query->whereNotNull('sent_to_zatca_at');
        }

        $invoices = $query->get();

        if ($invoices->isEmpty()) {
            $this->error('❌ لم يتم العثور على فواتير للتحديث');
            return 1;
        }

        $this->info("📊 تم العثور على {$invoices->count()} فاتورة للتحديث");

        $updated = 0;
        $failed = 0;
        $skipped = 0;

        $progressBar = $this->output->createProgressBar($invoices->count());
        $progressBar->start();

        foreach ($invoices as $invoice) {
            try {
                $qrInfo = $invoice->getQrCodeInfo();
                
                // عرض معلومات الفاتورة في وضع المعاينة
                if ($dryRun) {
                    $this->newLine();
                    $this->line("📄 فاتورة #{$invoice->invoice_number}:");
                    $this->line("   - المرحلة الحالية: Phase {$qrInfo['phase']}");
                    $this->line("   - مرسلة لـ ZATCA: " . ($qrInfo['sent_to_zatca'] ? 'نعم' : 'لا'));
                    $this->line("   - تحتوي على توقيع: " . ($qrInfo['contains_signature'] ? 'نعم' : 'لا'));
                    $this->line("   - تحتوي على مفتاح عام: " . ($qrInfo['contains_public_key'] ? 'نعم' : 'لا'));
                    $this->line("   - تحتوي على طابع تشفيري: " . ($qrInfo['contains_cryptographic_stamp'] ? 'نعم' : 'لا'));
                    
                    if ($force || $qrInfo['sent_to_zatca']) {
                        $this->line("   ✅ سيتم تحديثها");
                    } else {
                        $this->line("   ⏭️  سيتم تخطيها (غير مرسلة لـ ZATCA)");
                    }
                    continue;
                }

                // تخطي الفواتير غير المرسلة إذا لم يتم استخدام force
                if (!$force && !$qrInfo['sent_to_zatca']) {
                    $skipped++;
                    $progressBar->advance();
                    continue;
                }

                // تحديث QR Code
                if ($invoice->updateQrCodeForPhase2()) {
                    $updated++;
                } else {
                    $failed++;
                }

            } catch (\Exception $e) {
                $failed++;
                Log::error("فشل في تحديث QR Code للفاتورة {$invoice->id}", [
                    'error' => $e->getMessage()
                ]);
            }

            $progressBar->advance();
        }

        $progressBar->finish();
        $this->newLine();

        if ($dryRun) {
            $this->info('🔍 انتهت المعاينة. استخدم الأمر بدون --dry-run لتطبيق التغييرات');
        } else {
            $this->info('✅ انتهى التحديث!');
            $this->table(
                ['الحالة', 'العدد'],
                [
                    ['تم التحديث بنجاح', $updated],
                    ['فشل في التحديث', $failed],
                    ['تم التخطي', $skipped],
                    ['المجموع', $invoices->count()],
                ]
            );

            if ($failed > 0) {
                $this->warn("⚠️  فشل في تحديث {$failed} فاتورة. راجع ملف السجل للتفاصيل");
            }
        }

        return 0;
    }
}