<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\ZatcaService;
use App\Models\Invoice;
use App\Models\Company;
use App\Models\ZatcaSettings;
use Carbon\Carbon;

class TestZatcaFixes extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'zatca:test-fixes {--invoice=} {--comprehensive}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test all ZATCA fixes comprehensively';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 بدء اختبار إصلاحات ZATCA الشاملة...');
        $this->newLine();

        try {
            // الحصول على إعدادات ZATCA
            $settings = ZatcaSettings::first();
            if (!$settings) {
                $this->error('❌ إعدادات ZATCA غير موجودة. يرجى إنشاء إعدادات ZATCA أولاً.');
                return 1;
            }

            $zatcaService = new ZatcaService($settings);

            // إنشاء فاتورة اختبار أو استخدام فاتورة موجودة
            $invoice = $this->getTestInvoice();
            
            if (!$invoice) {
                $this->error('❌ فشل في إنشاء أو العثور على فاتورة للاختبار.');
                return 1;
            }

            $this->info("📄 اختبار الفاتورة: {$invoice->invoice_number}");
            $this->info("🏢 نوع الفاتورة: {$invoice->type}");
            $this->newLine();

            // تشغيل الاختبارات الشاملة
            if ($this->option('comprehensive')) {
                $results = $zatcaService->testAllZatcaFixesComprehensive($invoice);
            } else {
                $results = $zatcaService->testZatcaComprehensiveFixes($invoice);
            }

            // عرض النتائج
            $this->displayTestResults($results);

            // اختبار توليد XML
            $this->info('🔧 اختبار توليد XML...');
            try {
                $xmlContent = $zatcaService->generateInvoiceXml($invoice);
                if (!empty($xmlContent)) {
                    $this->info('✅ تم توليد XML بنجاح');
                    $this->line("📏 حجم XML: " . strlen($xmlContent) . " bytes");
                } else {
                    $this->error('❌ فشل في توليد XML');
                }
            } catch (\Exception $e) {
                $this->error('❌ خطأ في توليد XML: ' . $e->getMessage());
            }

            $this->newLine();

            // اختبار توليد QR Code
            $this->info('🔍 اختبار توليد QR Code...');
            try {
                $qrCode = $zatcaService->generateQRCode($invoice);
                if (!empty($qrCode)) {
                    $this->info('✅ تم توليد QR Code بنجاح');
                    $this->line("📏 طول QR Code: " . strlen($qrCode) . " characters");
                    
                    // فحص تنسيق Base64
                    $decoded = base64_decode($qrCode, true);
                    if ($decoded !== false) {
                        $this->info('✅ تنسيق Base64 صحيح');
                        $this->line("📏 حجم البيانات المشفرة: " . strlen($decoded) . " bytes");
                    } else {
                        $this->error('❌ تنسيق Base64 غير صحيح');
                    }
                } else {
                    $this->error('❌ فشل في توليد QR Code');
                }
            } catch (\Exception $e) {
                $this->error('❌ خطأ في توليد QR Code: ' . $e->getMessage());
            }

            $this->newLine();

            // ملخص النتائج النهائي
            $this->displayFinalSummary($results);

            return $results['overall_status'] === 'passed' ? 0 : 1;

        } catch (\Exception $e) {
            $this->error('❌ خطأ عام في الاختبار: ' . $e->getMessage());
            $this->line('🔍 تفاصيل الخطأ: ' . $e->getTraceAsString());
            return 1;
        }
    }

    /**
     * الحصول على فاتورة اختبار
     */
    private function getTestInvoice(): ?Invoice
    {
        $invoiceId = $this->option('invoice');
        
        if ($invoiceId) {
            $invoice = Invoice::find($invoiceId);
            if ($invoice) {
                $invoice->load(['company', 'customer', 'items']);
                return $invoice;
            }
        }

        // إنشاء فاتورة اختبار
        return $this->createTestInvoice();
    }

    /**
     * إنشاء فاتورة اختبار
     */
    private function createTestInvoice(): ?Invoice
    {
        try {
            // البحث عن شركة موجودة أو إنشاء شركة اختبار
            $company = Company::first();
            if (!$company) {
                $this->info('📋 إنشاء شركة اختبار...');
                $company = Company::create([
                    'name' => 'شركة الاختبار المحدودة',
                    'vat_number' => '399999999900003',
                    'commercial_registration' => '1234567890',
                    'address' => 'شارع الملك عبدالعزيز',
                    'building_number' => '1234',
                    'district' => 'حي الملك فهد',
                    'city' => 'الرياض',
                    'postal_code' => '11564',
                    'phone' => '+966501234567',
                    'email' => 'test@company.com'
                ]);
            }

            $this->info('📋 إنشاء فاتورة اختبار...');
            
            $invoice = new Invoice([
                'invoice_number' => 'TEST-' . time(),
                'issue_date' => Carbon::now(),
                'type' => 'simplified',
                'subtotal' => 100.00,
                'tax_amount' => 15.00,
                'total_amount' => 115.00,
                'currency' => 'SAR',
                'notes' => 'فاتورة اختبار لفحص إصلاحات ZATCA',
                'company_id' => $company->id,
                'user_id' => 1, // افتراضي
                'uuid' => \Illuminate\Support\Str::uuid(),
                'invoice_counter' => 1,
                'supply_date' => Carbon::now()
            ]);

            // ربط الشركة
            $invoice->company = $company;

            // إنشاء عناصر وهمية
            $invoice->items = collect([
                (object)[
                    'name' => 'منتج اختبار',
                    'quantity' => 1,
                    'unit_price' => 100.00,
                    'line_total' => 100.00,
                    'tax_amount' => 15.00,
                    'tax_rate' => 15,
                    'is_taxable' => true,
                    'unit' => 'PCE',
                    'zatca_category_code' => 'S'
                ]
            ]);

            return $invoice;

        } catch (\Exception $e) {
            $this->error('❌ خطأ في إنشاء فاتورة الاختبار: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * عرض نتائج الاختبار
     */
    private function displayTestResults(array $results): void
    {
        $this->info('📊 نتائج الاختبارات:');
        $this->newLine();

        if (isset($results['overall_status'])) {
            $status = $results['overall_status'] === 'passed' ? '✅ نجح' : '❌ فشل';
            $this->line("📈 الحالة العامة: {$status}");
        }

        if (isset($results['summary'])) {
            $summary = $results['summary'];
            $this->line("📝 إجمالي الاختبارات: {$summary['total_tests']}");
            $this->line("✅ اختبارات ناجحة: {$summary['passed_tests']}");
            $this->line("❌ اختبارات فاشلة: {$summary['failed_tests']}");
        }

        $this->newLine();

        // عرض تفاصيل الاختبارات
        if (isset($results['tests']) && is_array($results['tests'])) {
            $this->info('🔍 تفاصيل الاختبارات:');
            
            foreach ($results['tests'] as $testName => $testResult) {
                $status = $testResult['passed'] ? '✅' : '❌';
                $testDisplayName = $testResult['test_name'] ?? $testName;
                $this->line("{$status} {$testDisplayName}");
                
                if (isset($testResult['details']) && is_array($testResult['details'])) {
                    foreach ($testResult['details'] as $detail) {
                        $this->line("   📋 {$detail}");
                    }
                }
                
                if (isset($testResult['errors']) && is_array($testResult['errors']) && !empty($testResult['errors'])) {
                    foreach ($testResult['errors'] as $error) {
                        $this->line("   ⚠️  {$error}");
                    }
                }
            }
        }

        $this->newLine();

        // عرض حالة الإصلاحات
        if (isset($results['fixes_validation'])) {
            $this->info('🔧 حالة الإصلاحات:');
            $fixes = $results['fixes_validation'];
            
            $fixNames = [
                'signature_error_fixed' => 'إصلاح خطأ التوقيع (SIGNATURE_ERROR)',
                'qrcode_invalid_fixed' => 'إصلاح QR Code (QRCODE_INVALID)',
                'xsd_schema_error_fixed' => 'إصلاح مخطط XML (XSD_SCHEMA_ERROR)',
                'br_ksa_17_fixed' => 'إصلاح BR-KSA-17',
                'null_pointer_exception_fixed' => 'إصلاح NullPointerException'
            ];
            
            foreach ($fixNames as $key => $name) {
                if (isset($fixes[$key])) {
                    $status = $fixes[$key] ? '✅' : '❌';
                    $this->line("{$status} {$name}");
                }
            }
        }

        $this->newLine();

        // عرض التوصيات
        if (isset($results['recommendations']) && is_array($results['recommendations']) && !empty($results['recommendations'])) {
            $this->info('💡 التوصيات:');
            foreach ($results['recommendations'] as $recommendation) {
                $this->line("   💭 {$recommendation}");
            }
            $this->newLine();
        }
    }

    /**
     * عرض الملخص النهائي
     */
    private function displayFinalSummary(array $results): void
    {
        $this->info('📋 الملخص النهائي:');
        $this->newLine();

        $overallStatus = $results['overall_status'] ?? 'unknown';
        
        if ($overallStatus === 'passed') {
            $this->info('🎉 تهانينا! جميع اختبارات ZATCA نجحت بنجاح.');
            $this->info('✅ جميع الإصلاحات المطلوبة تعمل بشكل صحيح.');
            $this->info('🚀 النظام جاهز للإنتاج مع ZATCA.');
        } elseif ($overallStatus === 'failed') {
            $this->error('⚠️  بعض الاختبارات فشلت. يرجى مراجعة الأخطاء أعلاه.');
            $this->line('🔧 قم بإصلاح المشاكل المذكورة وأعد تشغيل الاختبار.');
        } else {
            $this->warn('❓ حالة غير محددة للاختبارات.');
        }

        $this->newLine();
        
        // نصائح الاستخدام
        $this->info('💡 نصائح للاستخدام:');
        $this->line('   🔍 استخدم --comprehensive للاختبار الشامل');
        $this->line('   📄 استخدم --invoice=ID لاختبار فاتورة محددة');
        $this->line('   📝 راجع ملفات الـ Log للحصول على تفاصيل إضافية');
        
        $this->newLine();
        $this->info('تم الانتهاء من الاختبار! 🏁');
    }
}