<?php

namespace App\Console\Commands;

use App\Jobs\SyncZatcaInvoicesJob;
use App\Models\ZatcaSettings;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class SyncZatcaInvoicesCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'zatca:sync-invoices 
                            {--company-id= : معرف الشركة المحدد}
                            {--environment= : البيئة (sandbox أو production)}
                            {--force : تشغيل المزامنة حتى لو لم تكن مطلوبة}
                            {--background : تشغيل المزامنة في الخلفية}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'مزامنة الفواتير مع نظام ZATCA';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 بدء عملية مزامنة فواتير ZATCA...');

        try {
            // الحصول على إعدادات ZATCA النشطة
            $query = ZatcaSettings::where('is_active', true);

            if ($this->option('company-id')) {
                $query->where('company_id', $this->option('company-id'));
            }

            if ($this->option('environment')) {
                $query->where('environment', $this->option('environment'));
            }

            $zatcaSettings = $query->get();

            if ($zatcaSettings->isEmpty()) {
                $this->error('❌ لم يتم العثور على إعدادات ZATCA نشطة');
                return Command::FAILURE;
            }

            $this->info("📋 تم العثور على {$zatcaSettings->count()} إعداد ZATCA");

            $totalSynced = 0;

            foreach ($zatcaSettings as $settings) {
                $this->line("🏢 معالجة الشركة: {$settings->company->name}");

                // التحقق من صحة الإعدادات
                if (!$settings->isValid()) {
                    $this->warn("⚠️  إعدادات ZATCA غير صالحة للشركة: {$settings->company->name}");
                    continue;
                }

                // التحقق من ضرورة المزامنة
                if (!$this->option('force') && !$this->shouldSync($settings)) {
                    $this->line("ℹ️  لا تحتاج الشركة {$settings->company->name} للمزامنة في الوقت الحالي");
                    continue;
                }

                if ($this->option('background')) {
                    // تشغيل المزامنة في الخلفية
                    SyncZatcaInvoicesJob::dispatch($settings->id);
                    $this->info("🔄 تم إرسال مهمة المزامنة للخلفية للشركة: {$settings->company->name}");
                } else {
                    // تشغيل المزامنة مباشرة
                    $results = $this->syncCompanyInvoices($settings);
                    $totalSynced += $results['success'];
                    
                    $this->displayResults($settings->company->name, $results);
                }
            }

            if (!$this->option('background')) {
                $this->info("✅ تمت المزامنة بنجاح - إجمالي الفواتير المُزامنة: {$totalSynced}");
            } else {
                $this->info("✅ تم إرسال جميع مهام المزامنة للخلفية");
            }

            return Command::SUCCESS;

        } catch (\Exception $e) {
            $this->error("❌ خطأ في عملية المزامنة: {$e->getMessage()}");
            Log::error('ZATCA sync command failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return Command::FAILURE;
        }
    }

    /**
     * التحقق من ضرورة المزامنة
     */
    private function shouldSync(ZatcaSettings $settings): bool
    {
        // التحقق من آخر مزامنة
        $lastSync = $settings->operation_log ? 
            collect($settings->operation_log)->where('operation', 'sync')->last() : null;

        if (!$lastSync) {
            return true; // لم تتم مزامنة من قبل
        }

        $lastSyncTime = \Carbon\Carbon::parse($lastSync['timestamp']);
        
        // مزامنة كل ساعة كحد أدنى
        return $lastSyncTime->diffInHours(now()) >= 1;
    }

    /**
     * تنفيذ مزامنة فواتير الشركة
     */
    private function syncCompanyInvoices(ZatcaSettings $settings): array
    {
        $zatcaService = new \App\Services\ZatcaService($settings);
        
        // الحصول على الفواتير المرسلة خلال الأسبوع الماضي
        $invoices = \App\Models\Invoice::where('company_id', $settings->company_id)
            ->whereNotNull('sent_to_zatca_at')
            ->where('sent_to_zatca_at', '>', now()->subDays(7))
            ->get();

        if ($invoices->isEmpty()) {
            return ['total' => 0, 'success' => 0, 'failed' => 0, 'errors' => []];
        }

        $this->line("📊 معالجة {$invoices->count()} فاتورة...");

        $results = $zatcaService->syncMultipleInvoices($invoices);
        
        return [
            'total' => $invoices->count(),
            'success' => $results['success'],
            'failed' => $results['failed'],
            'errors' => $results['errors']
        ];
    }

    /**
     * عرض نتائج المزامنة
     */
    private function displayResults(string $companyName, array $results): void
    {
        $this->line("📈 نتائج الشركة {$companyName}:");
        $this->line("   • إجمالي الفواتير: {$results['total']}");
        
        if ($results['success'] > 0) {
            $this->line("   • <fg=green>نجح: {$results['success']}</>");
        }
        
        if ($results['failed'] > 0) {
            $this->line("   • <fg=red>فشل: {$results['failed']}</>");
            
            if (!empty($results['errors'])) {
                $this->line("   📝 الأخطاء:");
                foreach (array_slice($results['errors'], 0, 3) as $error) {
                    $this->line("      - {$error['invoice_number']}: {$error['error']}");
                }
                
                if (count($results['errors']) > 3) {
                    $remaining = count($results['errors']) - 3;
                    $this->line("      ... و {$remaining} أخطاء أخرى");
                }
            }
        }
        
        $this->line('');
    }
}