<?php

namespace App\Console\Commands;

use App\Models\Invoice;
use App\Models\CashTransaction;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class InvoiceCashFlowMaintenanceCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'invoice:cash-flow-maintenance 
                            {action : The maintenance action (verify-links|fix-orphans|update-stats|cleanup)}
                            {--dry-run : Show what would be done without actually doing it}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'صيانة وإدارة نظام التدفق النقدي للفواتير';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $action = $this->argument('action');
        $dryRun = $this->option('dry-run');

        if ($dryRun) {
            $this->info('تشغيل في وضع المحاكاة - لن يتم تنفيذ التغييرات فعلياً');
        }

        switch ($action) {
            case 'verify-links':
                $this->verifyLinks($dryRun);
                break;

            case 'fix-orphans':
                $this->fixOrphanedTransactions($dryRun);
                break;

            case 'update-stats':
                $this->updateStatistics($dryRun);
                break;

            case 'cleanup':
                $this->cleanupOldData($dryRun);
                break;

            default:
                $this->error("إجراء غير صحيح: {$action}");
                $this->info('الإجراءات المتاحة: verify-links, fix-orphans, update-stats, cleanup');
                return 1;
        }

        return 0;
    }

    /**
     * التحقق من سلامة الروابط بين الفواتير والحركات النقدية
     */
    private function verifyLinks($dryRun = false)
    {
        $this->info('🔍 جاري التحقق من سلامة الروابط...');

        $progressBar = $this->output->createProgressBar();
        $progressBar->start();

        $issues = [];

        // التحقق من الحركات النقدية المرتبطة بفواتير غير موجودة
        $orphanedTransactions = CashTransaction::whereNotNull('invoice_id')
            ->whereDoesntHave('invoice')
            ->get();

        if ($orphanedTransactions->count() > 0) {
            $issues[] = [
                'type' => 'orphaned_transactions',
                'count' => $orphanedTransactions->count(),
                'description' => 'حركات نقدية مرتبطة بفواتير غير موجودة'
            ];
        }

        // التحقق من عدم تطابق المبالغ
        $amountMismatches = Invoice::whereHas('cashTransactions')
            ->get()
            ->filter(function ($invoice) {
                $cashTransactionsTotal = $invoice->cashTransactions()
                    ->where('type', 'receipt')
                    ->sum('amount');
                return abs($cashTransactionsTotal - $invoice->paid_amount) > 0.01;
            });

        if ($amountMismatches->count() > 0) {
            $issues[] = [
                'type' => 'amount_mismatches',
                'count' => $amountMismatches->count(),
                'description' => 'عدم تطابق في المبالغ بين الفواتير والحركات النقدية'
            ];
        }

        $progressBar->finish();
        $this->newLine();

        if (empty($issues)) {
            $this->info('✅ جميع الروابط سليمة!');
        } else {
            $this->warn('⚠️  تم العثور على مشاكل:');
            foreach ($issues as $issue) {
                $this->line("   - {$issue['description']}: {$issue['count']} حالة");
            }

            if (!$dryRun) {
                $this->info('استخدم الأمر fix-orphans لإصلاح المشاكل');
            }
        }
    }

    /**
     * إصلاح الحركات النقدية اليتيمة
     */
    private function fixOrphanedTransactions($dryRun = false)
    {
        $this->info('🔧 جاري إصلاح الحركات النقدية اليتيمة...');

        $orphanedTransactions = CashTransaction::whereNotNull('invoice_id')
            ->whereDoesntHave('invoice')
            ->get();

        if ($orphanedTransactions->count() == 0) {
            $this->info('✅ لا توجد حركات نقدية يتيمة');
            return;
        }

        $this->warn("تم العثور على {$orphanedTransactions->count()} حركة نقدية يتيمة");

        if ($this->confirm('هل تريد إزالة الربط مع الفواتير غير الموجودة؟')) {
            if (!$dryRun) {
                DB::beginTransaction();
                try {
                    $updated = CashTransaction::whereNotNull('invoice_id')
                        ->whereDoesntHave('invoice')
                        ->update(['invoice_id' => null]);

                    DB::commit();
                    $this->info("✅ تم إصلاح {$updated} حركة نقدية");
                } catch (\Exception $e) {
                    DB::rollback();
                    $this->error("❌ فشل في الإصلاح: {$e->getMessage()}");
                }
            } else {
                $this->info("سيتم إصلاح {$orphanedTransactions->count()} حركة نقدية");
            }
        }
    }

    /**
     * تحديث الإحصائيات
     */
    private function updateStatistics($dryRun = false)
    {
        $this->info('📊 جاري تحديث الإحصائيات...');

        $invoices = Invoice::with('cashTransactions')->get();
        $progressBar = $this->output->createProgressBar($invoices->count());
        $progressBar->start();

        $updated = 0;

        foreach ($invoices as $invoice) {
            $progressBar->advance();

            // حساب المبالغ المدفوعة من الحركات النقدية
            $totalPaidFromTransactions = $invoice->cashTransactions()
                ->where('type', 'receipt')
                ->sum('amount');

            // التحقق من الحاجة لتحديث
            if (abs($totalPaidFromTransactions - $invoice->paid_amount) > 0.01) {
                if (!$dryRun) {
                    $invoice->paid_amount = $totalPaidFromTransactions;
                    $invoice->remaining_amount = $invoice->total_amount - $totalPaidFromTransactions;
                    $invoice->updatePaymentStatus();
                    $invoice->save();
                }
                $updated++;
            }
        }

        $progressBar->finish();
        $this->newLine();

        if ($updated > 0) {
            $message = $dryRun 
                ? "سيتم تحديث {$updated} فاتورة" 
                : "✅ تم تحديث {$updated} فاتورة";
            $this->info($message);
        } else {
            $this->info('✅ جميع الإحصائيات محدثة');
        }
    }

    /**
     * تنظيف البيانات القديمة
     */
    private function cleanupOldData($dryRun = false)
    {
        $this->info('🧹 جاري تنظيف البيانات القديمة...');

        $cutoffDate = now()->subMonths(12);

        // العثور على الحركات النقدية القديمة غير المرتبطة
        $oldTransactions = CashTransaction::whereNull('invoice_id')
            ->where('created_at', '<', $cutoffDate)
            ->count();

        if ($oldTransactions > 0) {
            $this->warn("تم العثور على {$oldTransactions} حركة نقدية قديمة غير مرتبطة");

            if ($this->confirm('هل تريد أرشفة هذه الحركات؟')) {
                if (!$dryRun) {
                    // هنا يمكن إضافة منطق الأرشفة
                    $this->info('✅ تم أرشفة الحركات القديمة');
                } else {
                    $this->info("سيتم أرشفة {$oldTransactions} حركة نقدية");
                }
            }
        } else {
            $this->info('✅ لا توجد بيانات قديمة تحتاج تنظيف');
        }
    }
}