<?php

namespace App\Console\Commands;

use App\Models\Supplier;
use App\Models\PurchaseInvoice;
use App\Models\SupplierDebtSchedule;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class InitializeProcurementSystem extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'procurement:initialize 
                          {--migrate : Run migrations first}
                          {--seed : Seed sample data}
                          {--update-schedules : Update debt schedules}
                          {--calculate-scores : Calculate performance scores}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Initialize and update the procurement management system';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 تهيئة نظام إدارة المشتريات والموردين...');

        // تشغيل الـ Migrations
        if ($this->option('migrate')) {
            $this->info('📊 تشغيل قاعدة البيانات...');
            $this->call('migrate');
        }

        // إنشاء الجداول المطلوبة
        $this->createSupplierDebtSchedules();
        
        // حساب نقاط الأداء
        if ($this->option('calculate-scores')) {
            $this->calculatePerformanceScores();
        }

        // تحديث جدولة الديون
        if ($this->option('update-schedules')) {
            $this->updateDebtSchedules();
        }

        // إنشاء بيانات تجريبية
        if ($this->option('seed')) {
            $this->seedSampleData();
        }

        $this->info('✅ تم تهيئة النظام بنجاح!');
        $this->showSystemStats();
    }

    /**
     * إنشاء جدولة الديون للفواتير المستحقة
     */
    private function createSupplierDebtSchedules()
    {
        $this->info('📋 إنشاء جدولة الديون...');

        $outstandingInvoices = PurchaseInvoice::where('payment_status', '!=', 'paid')
            ->where('status', '!=', 'cancelled')
            ->where('remaining_amount', '>', 0)
            ->whereDoesntHave('debtSchedules')
            ->get();

        $count = 0;
        foreach ($outstandingInvoices as $invoice) {
            SupplierDebtSchedule::createScheduleForInvoice($invoice);
            $count++;
        }

        $this->info("📅 تم إنشاء {$count} جدولة دفع جديدة");
    }

    /**
     * حساب نقاط أداء الموردين
     */
    private function calculatePerformanceScores()
    {
        $this->info('🎯 حساب نقاط أداء الموردين...');

        $suppliers = Supplier::with(['purchaseInvoices', 'purchaseOrders'])->get();
        $bar = $this->output->createProgressBar($suppliers->count());

        foreach ($suppliers as $supplier) {
            $score = $this->calculateSupplierScore($supplier);
            $riskLevel = $this->calculateRiskLevel($supplier);
            
            $supplier->update([
                'performance_score' => $score,
                'risk_level' => $riskLevel,
            ]);

            $bar->advance();
        }

        $bar->finish();
        $this->line('');
        $this->info('✅ تم تحديث نقاط الأداء لجميع الموردين');
    }

    /**
     * تحديث جدولة الديون المتأخرة
     */
    private function updateDebtSchedules()
    {
        $this->info('⏰ تحديث جدولة الديون المتأخرة...');

        // تحديد الجدولات المتأخرة
        $overdueSchedules = SupplierDebtSchedule::where('schedule_date', '<', now())
            ->where('status', '!=', 'paid')
            ->where('status', '!=', 'overdue')
            ->get();

        foreach ($overdueSchedules as $schedule) {
            $schedule->markAsOverdue();
        }

        // إرسال تذكيرات للجدولات المستحقة قريباً
        $dueSoonSchedules = SupplierDebtSchedule::dueSoon(3)
            ->where('status', 'scheduled')
            ->get();

        foreach ($dueSoonSchedules as $schedule) {
            // يمكن إضافة منطق إرسال التذكيرات هنا
        }

        $this->info("⚠️  تم تحديث {$overdueSchedules->count()} جدولة متأخرة");
        $this->info("📢 {$dueSoonSchedules->count()} تذكير مستحق قريباً");
    }

    /**
     * إنشاء بيانات تجريبية
     */
    private function seedSampleData()
    {
        if (!$this->confirm('هل تريد إنشاء بيانات تجريبية؟ (سيتم حذف البيانات الموجودة)')) {
            return;
        }

        $this->info('🌱 إنشاء بيانات تجريبية...');

        // إنشاء موردين تجريبيين
        $suppliers = [
            [
                'name' => 'شركة التوريدات المتقدمة',
                'email' => 'info@advanced-supplies.com',
                'phone' => '0112345678',
                'credit_limit' => 100000,
                'penalty_rate' => 0.02,
                'delivery_days' => 7,
            ],
            [
                'name' => 'مؤسسة الجودة للتجارة',
                'email' => 'sales@quality-trade.com',
                'phone' => '0112345679',
                'credit_limit' => 150000,
                'penalty_rate' => 0.015,
                'delivery_days' => 10,
            ],
        ];

        foreach ($suppliers as $supplierData) {
            Supplier::create($supplierData);
        }

        $this->info('✅ تم إنشاء البيانات التجريبية');
    }

    /**
     * حساب نقاط أداء المورد
     */
    private function calculateSupplierScore(Supplier $supplier): float
    {
        // معدل التسليم في الوقت (30%)
        $onTimeRate = $this->getOnTimeDeliveryRate($supplier);
        
        // جودة المنتجات (25%)
        $qualityRate = $this->getQualityRate($supplier);
        
        // الالتزام بالمدفوعات (25%)
        $paymentCompliance = $this->getPaymentCompliance($supplier);
        
        // التواصل والاستجابة (20%)
        $responsiveness = $this->getResponsiveness($supplier);

        $score = ($onTimeRate * 0.30) + 
                ($qualityRate * 0.25) + 
                ($paymentCompliance * 0.25) + 
                ($responsiveness * 0.20);

        return round($score, 2);
    }

    /**
     * تحديد مستوى المخاطر للمورد
     */
    private function calculateRiskLevel(Supplier $supplier): string
    {
        $riskScore = 0;

        // تجاوز حد الائتمان
        if ($supplier->exceedsCreditLimit()) {
            $riskScore += 30;
        }

        // وجود فواتير متأخرة
        $overdueAmount = $supplier->getOverdueAmount();
        if ($overdueAmount > 0) {
            $riskScore += 20;
            if ($overdueAmount > $supplier->credit_limit * 0.5) {
                $riskScore += 20;
            }
        }

        // نقاط الأداء منخفضة
        if ($supplier->performance_score < 60) {
            $riskScore += 30;
        } elseif ($supplier->performance_score < 75) {
            $riskScore += 15;
        }

        if ($riskScore >= 70) return 'critical';
        if ($riskScore >= 50) return 'high';
        if ($riskScore >= 25) return 'medium';
        return 'low';
    }

    /**
     * حساب معدل التسليم في الوقت
     */
    private function getOnTimeDeliveryRate(Supplier $supplier): float
    {
        $completedOrders = $supplier->purchaseOrders()
            ->where('status', 'completed')
            ->count();

        if ($completedOrders == 0) return 85; // افتراضي

        $onTimeOrders = $supplier->purchaseOrders()
            ->where('status', 'completed')
            ->whereColumn('actual_delivery_date', '<=', 'expected_delivery_date')
            ->count();

        return ($onTimeOrders / $completedOrders) * 100;
    }

    /**
     * حساب معدل الجودة
     */
    private function getQualityRate(Supplier $supplier): float
    {
        // افتراضي - يمكن تحسينه بناء على بيانات فعلية
        return 92;
    }

    /**
     * حساب معدل الامتثال للمدفوعات
     */
    private function getPaymentCompliance(Supplier $supplier): float
    {
        $totalInvoices = $supplier->purchaseInvoices()
            ->where('status', '!=', 'cancelled')
            ->count();

        if ($totalInvoices == 0) return 100;

        $paidOnTime = $supplier->purchaseInvoices()
            ->where('payment_status', 'paid')
            ->whereColumn('due_date', '>=', 'updated_at')
            ->count();

        return ($paidOnTime / $totalInvoices) * 100;
    }

    /**
     * حساب معدل الاستجابة
     */
    private function getResponsiveness(Supplier $supplier): float
    {
        // افتراضي - يمكن تحسينه بناء على بيانات فعلية
        return 88;
    }

    /**
     * عرض إحصائيات النظام
     */
    private function showSystemStats()
    {
        $this->line('');
        $this->info('📊 إحصائيات النظام:');
        $this->line('─────────────────────────────');
        
        $this->line('الموردين: ' . Supplier::count());
        $this->line('الموردين النشطين: ' . Supplier::active()->count());
        $this->line('إجمالي المديونيات: ' . number_format(Supplier::sum('current_balance'), 2) . ' ر.س');
        $this->line('الفواتير المعلقة: ' . PurchaseInvoice::where('payment_status', '!=', 'paid')->count());
        $this->line('الجدولات المتأخرة: ' . SupplierDebtSchedule::overdue()->count());
        
        // توزيع مستوى المخاطر
        $riskDistribution = Supplier::selectRaw('risk_level, COUNT(*) as count')
            ->groupBy('risk_level')
            ->pluck('count', 'risk_level')
            ->toArray();
            
        $this->line('');
        $this->info('توزيع مستوى المخاطر:');
        foreach ($riskDistribution as $level => $count) {
            $this->line("  {$level}: {$count}");
        }
    }
}