<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\SavedReport;
use App\Models\AuditLog;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class GenerateScheduledReports extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'accounting:generate-reports 
                            {--force : توليد جميع التقارير المجدولة حتى لو لم تكن مستحقة}
                            {--report= : توليد تقرير محدد بالمعرف}
                            {--dry-run : عرض التقارير المستحقة دون توليد فعلي}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'توليد التقارير المجدولة المستحقة';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $force = $this->option('force');
        $reportId = $this->option('report');
        $dryRun = $this->option('dry-run');

        $this->info('📊 بدء توليد التقارير المجدولة...');
        $this->info('📅 التاريخ والوقت: ' . now()->format('Y-m-d H:i:s'));
        $this->line('');

        if ($dryRun) {
            $this->warn('⚠️  وضع التجربة - عرض التقارير المستحقة فقط');
            $this->line('');
        }

        if ($reportId) {
            $this->generateSingleReport($reportId, $dryRun);
        } else {
            $this->generateDueReports($force, $dryRun);
        }

        $this->info('');
        $this->info('✅ تم إكمال عملية توليد التقارير');
        
        return 0;
    }

    /**
     * Generate a single report by ID
     */
    private function generateSingleReport($reportId, $dryRun)
    {
        $this->info("🎯 توليد التقرير المحدد (ID: {$reportId})...");

        $report = SavedReport::find($reportId);
        if (!$report) {
            $this->error("❌ التقرير غير موجود: {$reportId}");
            return;
        }

        if (!$report->is_active) {
            $this->warn("⚠️  التقرير غير نشط: {$report->name}");
            return;
        }

        $this->line("📋 اسم التقرير: {$report->name}");
        $this->line("📊 نوع التقرير: {$report->report_type_description}");
        $this->line("📄 التنسيق: {$report->format}");

        if (!$dryRun) {
            try {
                $filePath = $report->generate();
                $this->info("✅ تم توليد التقرير بنجاح");
                $this->line("📁 مسار الملف: {$filePath}");
                
                if (!empty($report->recipients)) {
                    $this->sendReportToRecipients($report);
                }

            } catch (\Exception $e) {
                $this->error("❌ خطأ في توليد التقرير: " . $e->getMessage());
                Log::error('Single report generation failed', [
                    'report_id' => $reportId,
                    'error' => $e->getMessage()
                ]);
            }
        } else {
            $this->line("سيتم توليد هذا التقرير");
        }
    }

    /**
     * Generate all due reports
     */
    private function generateDueReports($force, $dryRun)
    {
        $this->info('🔄 البحث عن التقارير المستحقة...');

        $query = SavedReport::scheduled()->active();
        
        if (!$force) {
            $query->dueForGeneration();
        }

        $dueReports = $query->with(['creator'])->get();

        if ($dueReports->isEmpty()) {
            $this->info('✅ لا توجد تقارير مستحقة للتوليد');
            return;
        }

        $this->info("📊 عدد التقارير المستحقة: {$dueReports->count()}");
        $this->line('');

        // عرض قائمة التقارير
        $headers = ['ID', 'اسم التقرير', 'النوع', 'التكرار', 'آخر توليد', 'التوليد التالي', 'المنشئ'];
        $rows = [];

        foreach ($dueReports as $report) {
            $rows[] = [
                $report->id,
                $report->name,
                $report->report_type_description,
                $report->schedule_frequency_description,
                $report->last_generated_at ? $report->last_generated_at->format('Y-m-d H:i') : 'لم يتم',
                $report->next_generation_at ? $report->next_generation_at->format('Y-m-d H:i') : '',
                $report->creator->name ?? '',
            ];
        }

        $this->table($headers, $rows);
        $this->line('');

        if ($dryRun) {
            $this->info('🎭 وضع التجربة - التقارير أعلاه ستكون مستحقة للتوليد');
            return;
        }

        // تأكيد المتابعة
        if (!$force && !$this->confirm('هل تريد متابعة توليد هذه التقارير؟')) {
            $this->warn('تم إلغاء العملية');
            return;
        }

        // توليد التقارير
        $this->info('🚀 بدء توليد التقارير...');
        $this->line('');

        $generated = 0;
        $failed = 0;
        $progressBar = $this->output->createProgressBar($dueReports->count());
        $progressBar->start();

        foreach ($dueReports as $report) {
            try {
                $filePath = $report->generate();
                $generated++;
                
                // إرسال للمستلمين إذا كان محدد
                if (!empty($report->recipients)) {
                    $this->sendReportToRecipients($report);
                }

                $progressBar->advance();

            } catch (\Exception $e) {
                $failed++;
                Log::error('Scheduled report generation failed', [
                    'report_id' => $report->id,
                    'report_name' => $report->name,
                    'error' => $e->getMessage()
                ]);
                
                $progressBar->advance();
            }
        }

        $progressBar->finish();
        $this->line('');
        $this->line('');

        // عرض النتائج
        $this->info("✅ تم توليد {$generated} تقرير بنجاح");
        
        if ($failed > 0) {
            $this->warn("⚠️  فشل في توليد {$failed} تقرير");
        }

        // تسجيل في سجل المراجعة
        AuditLog::logCustomEvent(
            'scheduled_reports_generated',
            "تم توليد {$generated} تقرير مجدول",
            [
                'generated_count' => $generated,
                'failed_count' => $failed,
                'total_due' => $dueReports->count(),
                'forced' => $force
            ],
            ['scheduled_reports', 'automation']
        );
    }

    /**
     * Send report to recipients via email
     */
    private function sendReportToRecipients($report)
    {
        if (empty($report->recipients) || !$report->file_path) {
            return;
        }

        try {
            $this->line("📧 إرسال التقرير للمستلمين...");
            
            foreach ($report->recipients as $recipient) {
                // هنا يمكن إضافة منطق إرسال الإيميل
                // Mail::to($recipient)->send(new ReportGenerated($report));
                $this->line("  ✉️  {$recipient}");
            }

            $this->info("✅ تم إرسال التقرير لـ " . count($report->recipients) . " مستلم");

        } catch (\Exception $e) {
            $this->warn("⚠️  خطأ في إرسال التقرير: " . $e->getMessage());
            Log::warning('Report email sending failed', [
                'report_id' => $report->id,
                'recipients' => $report->recipients,
                'error' => $e->getMessage()
            ]);
        }
    }
}