<?php

namespace App\Console\Commands;

use App\Services\InvoiceCashFlowService;
use Illuminate\Console\Command;
use Carbon\Carbon;

class GenerateCashFlowReportCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'invoice:generate-cash-flow-report
                            {--period=monthly : Report period (daily|weekly|monthly|yearly)}
                            {--start-date= : Start date (Y-m-d format)}
                            {--end-date= : End date (Y-m-d format)}
                            {--format=csv : Export format (csv|pdf|json)}
                            {--output= : Output file path}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'توليد تقرير التدفق النقدي للفواتير';

    protected InvoiceCashFlowService $cashFlowService;

    public function __construct(InvoiceCashFlowService $cashFlowService)
    {
        parent::__construct();
        $this->cashFlowService = $cashFlowService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔄 جاري توليد تقرير التدفق النقدي...');

        // تحديد الفترة الزمنية
        [$startDate, $endDate] = $this->determinePeriod();

        $this->info("📅 فترة التقرير: من {$startDate} إلى {$endDate}");

        try {
            // توليد التقرير
            $reportData = $this->cashFlowService->analyzeCashFlow(
                $startDate,
                $endDate
            );

            // عرض الملخص
            $this->displaySummary($reportData['summary']);

            // تصدير التقرير
            $format = $this->option('format');
            $outputPath = $this->exportReport($reportData, $format, $startDate, $endDate);

            $this->info("✅ تم توليد التقرير بنجاح: {$outputPath}");

        } catch (\Exception $e) {
            $this->error("❌ فشل في توليد التقرير: {$e->getMessage()}");
            return 1;
        }

        return 0;
    }

    /**
     * تحديد فترة التقرير
     */
    private function determinePeriod(): array
    {
        $startDate = $this->option('start-date');
        $endDate = $this->option('end-date');

        if ($startDate && $endDate) {
            return [$startDate, $endDate];
        }

        $period = $this->option('period');
        $now = Carbon::now();

        switch ($period) {
            case 'daily':
                return [
                    $now->format('Y-m-d'),
                    $now->format('Y-m-d')
                ];

            case 'weekly':
                return [
                    $now->startOfWeek()->format('Y-m-d'),
                    $now->endOfWeek()->format('Y-m-d')
                ];

            case 'monthly':
                return [
                    $now->startOfMonth()->format('Y-m-d'),
                    $now->endOfMonth()->format('Y-m-d')
                ];

            case 'yearly':
                return [
                    $now->startOfYear()->format('Y-m-d'),
                    $now->endOfYear()->format('Y-m-d')
                ];

            default:
                return [
                    $now->startOfMonth()->format('Y-m-d'),
                    $now->endOfMonth()->format('Y-m-d')
                ];
        }
    }

    /**
     * عرض ملخص التقرير
     */
    private function displaySummary(array $summary): void
    {
        $this->newLine();
        $this->info('📊 ملخص التقرير:');
        $this->line('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

        $this->table(
            ['المؤشر', 'القيمة'],
            [
                ['إجمالي الفواتير', number_format($summary['total_invoices'])],
                ['إجمالي المبلغ', number_format($summary['total_amount'], 2) . ' ر.س'],
                ['إجمالي المحصل', number_format($summary['total_paid'], 2) . ' ر.س'],
                ['المقبوضات النقدية', number_format($summary['cash_received'], 2) . ' ر.س'],
                ['التحويلات البنكية', number_format($summary['bank_transfers'], 2) . ' ر.س'],
                ['دفعات البطاقات', number_format($summary['card_payments'], 2) . ' ر.س'],
                ['دفعات الشيكات', number_format($summary['check_payments'], 2) . ' ر.س'],
                ['نسبة التحصيل', $summary['payment_percentage'] . '%'],
                ['المبلغ المتبقي', number_format($summary['total_remaining'], 2) . ' ر.س'],
            ]
        );
    }

    /**
     * تصدير التقرير
     */
    private function exportReport(array $data, string $format, string $startDate, string $endDate): string
    {
        $outputPath = $this->option('output');

        if (!$outputPath) {
            $timestamp = now()->format('Y-m-d_H-i-s');
            $outputPath = storage_path("app/reports/cash_flow_report_{$timestamp}.{$format}");
        }

        // إنشاء المجلد إذا لم يكن موجوداً
        $directory = dirname($outputPath);
        if (!is_dir($directory)) {
            mkdir($directory, 0755, true);
        }

        switch ($format) {
            case 'csv':
                $this->exportToCsv($data, $outputPath, $startDate, $endDate);
                break;

            case 'json':
                $this->exportToJson($data, $outputPath);
                break;

            case 'pdf':
                $this->info('⚠️  تصدير PDF غير متوفر حالياً، سيتم التصدير كـ CSV');
                $this->exportToCsv($data, str_replace('.pdf', '.csv', $outputPath), $startDate, $endDate);
                $outputPath = str_replace('.pdf', '.csv', $outputPath);
                break;

            default:
                $this->exportToCsv($data, $outputPath, $startDate, $endDate);
        }

        return $outputPath;
    }

    /**
     * تصدير إلى CSV
     */
    private function exportToCsv(array $data, string $path, string $startDate, string $endDate): void
    {
        $csvData = $this->cashFlowService->exportCashFlowReport($data);

        // إضافة معلومات إضافية
        array_unshift($csvData, 
            ['تقرير التدفق النقدي للفواتير'],
            ['تم التوليد في: ' . now()->format('Y-m-d H:i:s')],
            ['فترة التقرير: من ' . $startDate . ' إلى ' . $endDate],
            ['']
        );

        $file = fopen($path, 'w');
        fprintf($file, chr(0xEF) . chr(0xBB) . chr(0xBF)); // UTF-8 BOM

        foreach ($csvData as $row) {
            fputcsv($file, $row);
        }

        fclose($file);
    }

    /**
     * تصدير إلى JSON
     */
    private function exportToJson(array $data, string $path): void
    {
        $jsonData = [
            'generated_at' => now()->toISOString(),
            'report_data' => $data,
            'metadata' => [
                'version' => '1.0',
                'source' => 'Invoice Cash Flow System',
            ]
        ];

        file_put_contents(
            $path, 
            json_encode($jsonData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
        );
    }
}