<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\AccountBalance;
use App\Models\Budget;
use App\Models\AuditLog;
use App\Models\SavedReport;
use App\Models\FiscalYear;
use App\Models\Account;
use App\Models\JournalEntry;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AccountingMaintenanceCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'accounting:maintenance
                            {operation=all : العملية المطلوبة (all, balances, budgets, audit-logs, reports, backup)}
                            {--force : تشغيل العملية بدون تأكيد}
                            {--dry-run : عرض ما سيتم تنفيذه دون تنفيذ فعلي}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'تشغيل عمليات الصيانة الدورية للنظام المحاسبي';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $operation = $this->argument('operation');
        $dryRun = $this->option('dry-run');
        $force = $this->option('force');

        $this->info('🔧 بدء عمليات صيانة النظام المحاسبي...');
        $this->info('📅 التاريخ: ' . now()->format('Y-m-d H:i:s'));
        $this->line('');

        if ($dryRun) {
            $this->warn('⚠️  وضع التجربة - لن يتم تنفيذ أي تغييرات فعلية');
            $this->line('');
        }

        switch ($operation) {
            case 'all':
                $this->runAllMaintenance($dryRun, $force);
                break;
            case 'balances':
                $this->updateAccountBalances($dryRun, $force);
                break;
            case 'budgets':
                $this->updateBudgets($dryRun, $force);
                break;
            case 'audit-logs':
                $this->cleanupAuditLogs($dryRun, $force);
                break;
            case 'reports':
                $this->cleanupReports($dryRun, $force);
                break;
            case 'backup':
                $this->createBackup($dryRun, $force);
                break;
            default:
                $this->error('❌ عملية غير مدعومة: ' . $operation);
                return 1;
        }

        $this->info('');
        $this->info('✅ تم إكمال عمليات الصيانة بنجاح');
        
        return 0;
    }

    /**
     * Run all maintenance operations
     */
    private function runAllMaintenance($dryRun, $force)
    {
        $this->info('🚀 تشغيل جميع عمليات الصيانة...');
        $this->line('');

        $operations = [
            'تحديث أرصدة الحسابات' => fn() => $this->updateAccountBalances($dryRun, $force),
            'تحديث الميزانيات' => fn() => $this->updateBudgets($dryRun, $force),
            'تنظيف سجل المراجعة' => fn() => $this->cleanupAuditLogs($dryRun, $force),
            'تنظيف التقارير القديمة' => fn() => $this->cleanupReports($dryRun, $force),
            'إنشاء نسخة احتياطية' => fn() => $this->createBackup($dryRun, $force),
        ];

        foreach ($operations as $name => $operation) {
            $this->info("📋 {$name}...");
            $operation();
            $this->line('');
        }
    }

    /**
     * Update account balances
     */
    private function updateAccountBalances($dryRun, $force)
    {
        $this->info('💰 تحديث أرصدة الحسابات...');

        $currentFiscalYear = FiscalYear::current()->first();
        if (!$currentFiscalYear) {
            $this->warn('⚠️  لا توجد سنة مالية حالية');
            return;
        }

        $accounts = Account::where('is_active', true)->count();
        $this->line("📊 عدد الحسابات النشطة: {$accounts}");

        if (!$dryRun) {
            if (!$force && !$this->confirm('هل تريد المتابعة مع تحديث الأرصدة؟')) {
                $this->warn('تم إلغاء العملية');
                return;
            }

            try {
                $calculated = AccountBalance::calculateAllBalances($currentFiscalYear->id);
                $this->info("✅ تم حساب {$calculated} رصيد حساب");
                
                // تسجيل في سجل المراجعة
                AuditLog::logCustomEvent(
                    'balance_calculation',
                    "تم حساب أرصدة {$calculated} حساب للسنة المالية {$currentFiscalYear->name}",
                    ['fiscal_year_id' => $currentFiscalYear->id, 'calculated_count' => $calculated],
                    ['maintenance', 'balance_calculation']
                );
                
            } catch (\Exception $e) {
                $this->error('❌ خطأ في حساب الأرصدة: ' . $e->getMessage());
                Log::error('Account balance calculation error', ['error' => $e->getMessage()]);
            }
        } else {
            $this->line("سيتم حساب {$accounts} رصيد حساب");
        }
    }

    /**
     * Update budgets
     */
    private function updateBudgets($dryRun, $force)
    {
        $this->info('📊 تحديث الميزانيات...');

        $activeBudgets = Budget::where('is_active', true)
            ->where('auto_calculate', true)
            ->count();

        $this->line("📈 عدد الميزانيات النشطة: {$activeBudgets}");

        if (!$dryRun) {
            if (!$force && !$this->confirm('هل تريد المتابعة مع تحديث الميزانيات؟')) {
                $this->warn('تم إلغاء العملية');
                return;
            }

            try {
                $updated = 0;
                $errors = 0;

                Budget::where('is_active', true)
                    ->where('auto_calculate', true)
                    ->chunk(50, function ($budgets) use (&$updated, &$errors) {
                        foreach ($budgets as $budget) {
                            try {
                                $budget->calculateActualAmount();
                                $updated++;
                            } catch (\Exception $e) {
                                $errors++;
                                Log::warning('Budget update error', [
                                    'budget_id' => $budget->id,
                                    'error' => $e->getMessage()
                                ]);
                            }
                        }
                    });

                $this->info("✅ تم تحديث {$updated} ميزانية");
                if ($errors > 0) {
                    $this->warn("⚠️  {$errors} ميزانية بها أخطاء");
                }

                // تسجيل في سجل المراجعة
                AuditLog::logCustomEvent(
                    'budget_update',
                    "تم تحديث {$updated} ميزانية",
                    ['updated_count' => $updated, 'error_count' => $errors],
                    ['maintenance', 'budget_update']
                );

            } catch (\Exception $e) {
                $this->error('❌ خطأ في تحديث الميزانيات: ' . $e->getMessage());
                Log::error('Budget update error', ['error' => $e->getMessage()]);
            }
        } else {
            $this->line("سيتم تحديث {$activeBudgets} ميزانية");
        }
    }

    /**
     * Cleanup old audit logs
     */
    private function cleanupAuditLogs($dryRun, $force)
    {
        $this->info('🗑️  تنظيف سجل المراجعة القديم...');

        $daysToKeep = config('accounting.audit_log_retention_days', 365);
        $cutoffDate = now()->subDays($daysToKeep);
        
        $oldLogsCount = AuditLog::where('created_at', '<', $cutoffDate)->count();
        $this->line("📊 سجلات أقدم من {$daysToKeep} يوم: {$oldLogsCount}");

        if ($oldLogsCount == 0) {
            $this->info('✅ لا توجد سجلات قديمة للحذف');
            return;
        }

        if (!$dryRun) {
            if (!$force && !$this->confirm("هل تريد حذف {$oldLogsCount} سجل قديم؟")) {
                $this->warn('تم إلغاء العملية');
                return;
            }

            try {
                $deleted = AuditLog::cleanupOldLogs($daysToKeep);
                $this->info("✅ تم حذف {$deleted} سجل قديم");

                // تسجيل في سجل المراجعة
                AuditLog::logCustomEvent(
                    'audit_log_cleanup',
                    "تم حذف {$deleted} سجل مراجعة قديم",
                    ['deleted_count' => $deleted, 'days_kept' => $daysToKeep],
                    ['maintenance', 'cleanup']
                );

            } catch (\Exception $e) {
                $this->error('❌ خطأ في تنظيف سجل المراجعة: ' . $e->getMessage());
                Log::error('Audit log cleanup error', ['error' => $e->getMessage()]);
            }
        } else {
            $this->line("سيتم حذف {$oldLogsCount} سجل قديم");
        }
    }

    /**
     * Cleanup old report files
     */
    private function cleanupReports($dryRun, $force)
    {
        $this->info('📄 تنظيف ملفات التقارير القديمة...');

        $daysToKeep = config('accounting.report_retention_days', 30);
        $cutoffDate = now()->subDays($daysToKeep);
        
        $oldReportsCount = SavedReport::where('last_generated_at', '<', $cutoffDate)
            ->whereNotNull('file_path')
            ->count();

        $this->line("📊 تقارير أقدم من {$daysToKeep} يوم: {$oldReportsCount}");

        if ($oldReportsCount == 0) {
            $this->info('✅ لا توجد ملفات تقارير قديمة للحذف');
            return;
        }

        if (!$dryRun) {
            if (!$force && !$this->confirm("هل تريد حذف {$oldReportsCount} ملف تقرير قديم؟")) {
                $this->warn('تم إلغاء العملية');
                return;
            }

            try {
                $cleaned = SavedReport::cleanupOldFiles($daysToKeep);
                $this->info("✅ تم حذف {$cleaned} ملف تقرير قديم");

                // تسجيل في سجل المراجعة
                AuditLog::logCustomEvent(
                    'report_cleanup',
                    "تم حذف {$cleaned} ملف تقرير قديم",
                    ['cleaned_count' => $cleaned, 'days_kept' => $daysToKeep],
                    ['maintenance', 'cleanup']
                );

            } catch (\Exception $e) {
                $this->error('❌ خطأ في تنظيف ملفات التقارير: ' . $e->getMessage());
                Log::error('Report cleanup error', ['error' => $e->getMessage()]);
            }
        } else {
            $this->line("سيتم حذف {$oldReportsCount} ملف تقرير قديم");
        }
    }

    /**
     * Create database backup
     */
    private function createBackup($dryRun, $force)
    {
        $this->info('💾 إنشاء نسخة احتياطية...');

        if (!$dryRun) {
            if (!$force && !$this->confirm('هل تريد إنشاء نسخة احتياطية للنظام المحاسبي؟')) {
                $this->warn('تم إلغاء العملية');
                return;
            }

            try {
                // تحديد المجلد والملف
                $backupDir = storage_path('app/backups/accounting');
                if (!is_dir($backupDir)) {
                    mkdir($backupDir, 0755, true);
                }

                $filename = 'accounting_backup_' . now()->format('Y-m-d_H-i-s') . '.sql';
                $filepath = $backupDir . '/' . $filename;

                // تصدير الجداول المحاسبية
                $accountingTables = [
                    'accounts', 'journal_entries', 'journal_entry_lines',
                    'cash_transactions', 'bank_accounts', 'bank_transactions',
                    'cost_centers', 'fiscal_years', 'budgets', 'account_balances',
                    'accounting_settings', 'saved_reports', 'audit_logs'
                ];

                $mysqldump = config('database.connections.mysql.dump_command_path', 'mysqldump');
                $database = config('database.connections.mysql.database');
                $username = config('database.connections.mysql.username');
                $password = config('database.connections.mysql.password');
                $host = config('database.connections.mysql.host');

                $tablesStr = implode(' ', $accountingTables);
                $command = "{$mysqldump} -h{$host} -u{$username} -p{$password} {$database} {$tablesStr} > {$filepath}";

                exec($command, $output, $returnCode);

                if ($returnCode === 0 && file_exists($filepath)) {
                    $fileSize = round(filesize($filepath) / 1024 / 1024, 2);
                    $this->info("✅ تم إنشاء النسخة الاحتياطية: {$filename} ({$fileSize} ميجابايت)");

                    // تسجيل في سجل المراجعة
                    AuditLog::logCustomEvent(
                        'backup_created',
                        "تم إنشاء نسخة احتياطية للنظام المحاسبي",
                        ['filename' => $filename, 'size_mb' => $fileSize],
                        ['maintenance', 'backup']
                    );

                } else {
                    $this->error('❌ فشل في إنشاء النسخة الاحتياطية');
                }

            } catch (\Exception $e) {
                $this->error('❌ خطأ في إنشاء النسخة الاحتياطية: ' . $e->getMessage());
                Log::error('Backup creation error', ['error' => $e->getMessage()]);
            }
        } else {
            $this->line('سيتم إنشاء نسخة احتياطية للنظام المحاسبي');
        }
    }
}