<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\AccountingService;
use App\Models\Account;
use App\Models\JournalEntry;
use App\Models\CashTransaction;
use App\Models\CostCenter;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class AccountingMaintenance extends Command
{
    /**
     * اسم ووصف الأمر
     */
    protected $signature = 'accounting:maintenance {action?} {--date=} {--force}';
    
    protected $description = 'تنفيذ عمليات الصيانة للنظام المحاسبي';

    protected $accountingService;

    public function __construct(AccountingService $accountingService)
    {
        parent::__construct();
        $this->accountingService = $accountingService;
    }

    /**
     * تنفيذ الأمر
     */
    public function handle()
    {
        $action = $this->argument('action') ?? 'all';
        
        $this->info('🚀 بدء عمليات صيانة النظام المحاسبي...');
        
        try {
            switch ($action) {
                case 'balances':
                    $this->updateAccountBalances();
                    break;
                    
                case 'cleanup':
                    $this->cleanupData();
                    break;
                    
                case 'validate':
                    $this->validateData();
                    break;
                    
                case 'backup':
                    $this->backupAccountingData();
                    break;
                    
                case 'reports':
                    $this->generateScheduledReports();
                    break;
                    
                case 'all':
                    $this->runAllMaintenance();
                    break;
                    
                default:
                    $this->error('❌ عملية غير معروفة: ' . $action);
                    $this->showAvailableActions();
                    return 1;
            }
            
            $this->info('✅ تم إنجاز عمليات الصيانة بنجاح!');
            return 0;
            
        } catch (\Exception $e) {
            $this->error('❌ خطأ في عمليات الصيانة: ' . $e->getMessage());
            Log::error('Accounting Maintenance Error', [
                'action' => $action,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return 1;
        }
    }

    /**
     * تحديث أرصدة الحسابات
     */
    private function updateAccountBalances()
    {
        $this->info('📊 تحديث أرصدة الحسابات...');
        
        $date = $this->option('date') ? Carbon::parse($this->option('date')) : now();
        
        $accounts = Account::where('is_active', true)->get();
        $progressBar = $this->output->createProgressBar($accounts->count());
        $progressBar->start();
        
        $updatedCount = 0;
        foreach ($accounts as $account) {
            $oldBalance = $account->current_balance;
            $newBalance = $this->accountingService->getAccountBalance($account->id, $date);
            
            if (abs($oldBalance - $newBalance) > 0.01) {
                $account->update(['current_balance' => $newBalance]);
                $updatedCount++;
            }
            
            $progressBar->advance();
        }
        
        $progressBar->finish();
        $this->newLine();
        $this->info("✅ تم تحديث {$updatedCount} حساب من أصل {$accounts->count()}");
    }

    /**
     * تنظيف البيانات
     */
    private function cleanupData()
    {
        $this->info('🧹 تنظيف البيانات...');
        
        // حذف القيود غير المعتمدة القديمة (أكثر من 30 يوم)
        $oldDrafts = JournalEntry::where('is_approved', false)
            ->where('created_at', '<', now()->subDays(30))
            ->get();
            
        if ($oldDrafts->count() > 0) {
            if ($this->option('force') || $this->confirm("هل تريد حذف {$oldDrafts->count()} قيد مسودة قديم؟")) {
                foreach ($oldDrafts as $draft) {
                    $draft->journalEntryLines()->delete();
                    $draft->delete();
                }
                $this->info("✅ تم حذف {$oldDrafts->count()} قيد مسودة قديم");
            }
        } else {
            $this->info('✅ لا توجد مسودات قديمة للحذف');
        }

        // تنظيف المعاملات النقدية غير المطابقة القديمة
        $oldUnreconciledTransactions = CashTransaction::where('is_reconciled', false)
            ->where('created_at', '<', now()->subDays(90))
            ->count();
            
        if ($oldUnreconciledTransactions > 0) {
            $this->warn("⚠️  يوجد {$oldUnreconciledTransactions} معاملة نقدية غير مطابقة قديمة");
            $this->info('💡 يُنصح بمراجعة هذه المعاملات ومطابقتها');
        }
    }

    /**
     * التحقق من صحة البيانات
     */
    private function validateData()
    {
        $this->info('🔍 التحقق من صحة البيانات...');
        
        $errors = [];
        
        // التحقق من توازن القيود المعتمدة
        $this->info('🔄 فحص توازن القيود المحاسبية...');
        $unbalancedEntries = JournalEntry::where('is_approved', true)->get()->filter(function ($entry) {
            return !$this->accountingService->validateJournalEntryBalance($entry->id);
        });
        
        if ($unbalancedEntries->count() > 0) {
            $errors[] = "يوجد {$unbalancedEntries->count()} قيد غير متوازن";
            foreach ($unbalancedEntries as $entry) {
                $this->error("❌ القيد {$entry->reference} غير متوازن");
            }
        } else {
            $this->info('✅ جميع القيود متوازنة');
        }

        // التحقق من الحسابات المعطلة التي بها حركة
        $this->info('🔄 فحص الحسابات المعطلة...');
        $inactiveAccountsWithBalance = Account::where('is_active', false)
            ->where('current_balance', '!=', 0)
            ->get();
            
        if ($inactiveAccountsWithBalance->count() > 0) {
            $errors[] = "يوجد {$inactiveAccountsWithBalance->count()} حساب معطل برصيد غير صفر";
            foreach ($inactiveAccountsWithBalance as $account) {
                $this->warn("⚠️  الحساب {$account->code} - {$account->name} معطل برصيد {$account->current_balance}");
            }
        }

        // التحقق من مراكز التكلفة المطلوبة
        $this->info('🔄 فحص مراكز التكلفة...');
        $entriesWithoutCostCenter = \DB::table('journal_entry_lines')
            ->join('accounts', 'journal_entry_lines.account_id', '=', 'accounts.id')
            ->where('accounts.requires_cost_center', true)
            ->whereNull('journal_entry_lines.cost_center_id')
            ->count();
            
        if ($entriesWithoutCostCenter > 0) {
            $errors[] = "يوجد {$entriesWithoutCostCenter} سطر قيد بدون مركز تكلفة مطلوب";
        }

        // عرض النتائج
        if (count($errors) > 0) {
            $this->error('❌ تم العثور على أخطاء:');
            foreach ($errors as $error) {
                $this->error('  • ' . $error);
            }
        } else {
            $this->info('✅ جميع البيانات صحيحة');
        }
    }

    /**
     * نسخ احتياطي للبيانات المحاسبية
     */
    private function backupAccountingData()
    {
        $this->info('💾 إنشاء نسخة احتياطية للبيانات المحاسبية...');
        
        $timestamp = now()->format('Y-m-d_H-i-s');
        $backupDir = storage_path('app/backups/accounting');
        
        if (!file_exists($backupDir)) {
            mkdir($backupDir, 0755, true);
        }

        // نسخ احتياطي للجداول المحاسبية
        $tables = [
            'accounts',
            'journal_entries', 
            'journal_entry_lines',
            'cash_transactions',
            'bank_accounts',
            'bank_transactions',
            'cost_centers'
        ];

        foreach ($tables as $table) {
            $filename = "{$backupDir}/{$table}_{$timestamp}.sql";
            $command = "mysqldump --user=" . env('DB_USERNAME') . 
                      " --password=" . env('DB_PASSWORD') . 
                      " --host=" . env('DB_HOST') . 
                      " " . env('DB_DATABASE') . 
                      " {$table} > {$filename}";
            
            exec($command, $output, $returnVar);
            
            if ($returnVar === 0) {
                $this->info("✅ تم حفظ {$table}");
            } else {
                $this->error("❌ فشل في حفظ {$table}");
            }
        }
        
        $this->info("💾 تم حفظ النسخة الاحتياطية في: {$backupDir}");
    }

    /**
     * إنشاء التقارير المجدولة
     */
    private function generateScheduledReports()
    {
        $this->info('📊 إنشاء التقارير المجدولة...');
        
        // تقرير يومي للرصيد النقدي
        $cashBalance = Account::where('is_cash_account', true)
            ->orWhere('is_bank_account', true)
            ->sum('current_balance');
            
        Log::info('Daily Cash Balance Report', [
            'date' => now()->toDateString(),
            'total_cash_balance' => $cashBalance
        ]);

        // تقرير أسبوعي لمراكز التكلفة
        if (now()->dayOfWeek === 0) { // الأحد
            $costCenters = CostCenter::where('is_active', true)->get();
            foreach ($costCenters as $costCenter) {
                $actualAmount = $costCenter->actual_amount ?? 0;
                $budgetAmount = $costCenter->budget_amount ?? 0;
                $variance = $budgetAmount > 0 ? (($actualAmount - $budgetAmount) / $budgetAmount) * 100 : 0;
                
                Log::info('Weekly Cost Center Report', [
                    'cost_center' => $costCenter->name,
                    'budget' => $budgetAmount,
                    'actual' => $actualAmount,
                    'variance_percentage' => $variance
                ]);
            }
        }

        $this->info('✅ تم إنشاء التقارير المجدولة');
    }

    /**
     * تنفيذ جميع عمليات الصيانة
     */
    private function runAllMaintenance()
    {
        $this->info('🔄 تنفيذ جميع عمليات الصيانة...');
        
        $this->updateAccountBalances();
        $this->newLine();
        
        $this->validateData();
        $this->newLine();
        
        $this->cleanupData();
        $this->newLine();
        
        $this->generateScheduledReports();
    }

    /**
     * عرض العمليات المتاحة
     */
    private function showAvailableActions()
    {
        $this->info('العمليات المتاحة:');
        $this->line('  balances  - تحديث أرصدة الحسابات');
        $this->line('  cleanup   - تنظيف البيانات القديمة');
        $this->line('  validate  - التحقق من صحة البيانات');
        $this->line('  backup    - إنشاء نسخة احتياطية');
        $this->line('  reports   - إنشاء التقارير المجدولة');
        $this->line('  all       - تنفيذ جميع العمليات');
        $this->newLine();
        $this->info('المعاملات:');
        $this->line('  --date=YYYY-MM-DD  تاريخ محدد للعمليات');
        $this->line('  --force            تنفيذ بدون تأكيد');
    }
}