/**
 * نظام المبيعات الاحترافي - JavaScript محسن ومطور
 * Professional Sales System - Enhanced & Optimized JavaScript
 */

// متغيرات عامة
let itemIndex = 0;
let animationTimeout;
let validationErrors = [];

// تهيئة النظام عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    initializeProfessionalSalesWindow();
    setupAdvancedFormValidation();
    initializeAnimations();
    calculatePaymentAmounts();
    updateFormProgress();
});

/**
 * تهيئة نافذة المبيعات الاحترافية
 */
function initializeProfessionalSalesWindow() {
    // إضافة تأثيرات التحميل المحسنة
    addProfessionalLoadingEffects();
    
    // تحسين تفاعل النماذج
    enhanceProfessionalForms();
    
    // تهيئة الاختصارات
    initializeKeyboardShortcuts();
    
    // تهيئة التلميحات المتقدمة
    initializeAdvancedTooltips();
    
    // تهيئة التحديث التلقائي
    initializeAutoSave();
}

/**
 * إضافة تأثيرات التحميل الاحترافية
 */
function addProfessionalLoadingEffects() {
    const cards = document.querySelectorAll('.professional-card');
    const headerSection = document.querySelector('.header-section');
    
    // تأثير التحميل للعنوان
    if (headerSection) {
        headerSection.style.opacity = '0';
        headerSection.style.transform = 'translateY(-30px)';
        
        setTimeout(() => {
            headerSection.style.transition = 'all 0.8s cubic-bezier(0.25, 0.46, 0.45, 0.94)';
            headerSection.style.opacity = '1';
            headerSection.style.transform = 'translateY(0)';
        }, 100);
    }
    
    // تأثير التحميل للبطاقات
    cards.forEach((card, index) => {
        card.style.opacity = '0';
        card.style.transform = 'translateY(40px) scale(0.95)';
        
        setTimeout(() => {
            card.style.transition = 'all 0.8s cubic-bezier(0.25, 0.46, 0.45, 0.94)';
            card.style.opacity = '1';
            card.style.transform = 'translateY(0) scale(1)';
        }, 200 + (index * 150));
    });
    
    // تأثير موجة التحميل
    createLoadingWaveEffect();
}

/**
 * إنشاء تأثير موجة التحميل
 */
function createLoadingWaveEffect() {
    const wave = document.createElement('div');
    wave.className = 'loading-wave';
    wave.innerHTML = `
        <div class="wave-line"></div>
        <div class="wave-line"></div>
        <div class="wave-line"></div>
    `;
    
    // إضافة CSS للموجة
    const waveStyles = document.createElement('style');
    waveStyles.textContent = `
        .loading-wave {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            z-index: 9999;
            opacity: 0;
            animation: waveAppear 0.5s ease-out;
        }
        
        .wave-line {
            height: 100%;
            background: linear-gradient(90deg, #667eea, #764ba2, #667eea);
            background-size: 200% 100%;
            animation: waveMove 2s linear infinite;
        }
        
        @keyframes waveAppear {
            to { opacity: 1; }
        }
        
        @keyframes waveMove {
            0% { background-position: 200% 0; }
            100% { background-position: -200% 0; }
        }
    `;
    
    document.head.appendChild(waveStyles);
    document.body.appendChild(wave);
    
    // إزالة الموجة بعد التحميل
    setTimeout(() => {
        wave.style.opacity = '0';
        setTimeout(() => {
            if (wave.parentNode) wave.remove();
        }, 500);
    }, 2000);
}

/**
 * تحسين النماذج الاحترافية
 */
function enhanceProfessionalForms() {
    const inputs = document.querySelectorAll('.form-control-professional');
    
    inputs.forEach(input => {
        // تأثيرات التركيز المحسنة
        input.addEventListener('focus', function() {
            this.parentElement.classList.add('field-focused');
            addAdvancedGlowEffect(this);
            playFocusSound();
        });
        
        input.addEventListener('blur', function() {
            this.parentElement.classList.remove('field-focused');
            removeAdvancedGlowEffect(this);
            validateFieldInstantly(this);
        });
        
        // تأثير الكتابة المحسن
        input.addEventListener('input', function() {
            addAdvancedTypingEffect(this);
            clearFieldError(this);
            updateFormProgress();
            
            // حفظ تلقائي
            if (this.value) {
                scheduleAutoSave();
            }
        });
        
        // دعم الاختصارات
        input.addEventListener('keydown', function(e) {
            handleKeyboardShortcuts(e, this);
        });
    });
    
    // تحسين القوائم المنسدلة
    const selects = document.querySelectorAll('select.form-control-professional');
    selects.forEach(select => {
        select.addEventListener('change', function() {
            addSelectionAnimation(this);
            updateFormProgress();
            validateFieldInstantly(this);
        });
    });
}

/**
 * إضافة تأثير التوهج المتقدم
 */
function addAdvancedGlowEffect(element) {
    element.style.boxShadow = '0 0 20px rgba(102, 126, 234, 0.4), 0 0 40px rgba(102, 126, 234, 0.1)';
    element.style.transform = 'translateY(-2px) scale(1.02)';
    element.style.borderColor = '#667eea';
    
    // تأثير الشعاع
    const beam = document.createElement('div');
    beam.className = 'focus-beam';
    beam.style.cssText = `
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 2px;
        background: linear-gradient(90deg, transparent, #667eea, transparent);
        animation: beamMove 1.5s ease-in-out infinite;
        pointer-events: none;
        z-index: 10;
    `;
    
    element.parentElement.style.position = 'relative';
    element.parentElement.appendChild(beam);
}

/**
 * إزالة تأثير التوهج المتقدم
 */
function removeAdvancedGlowEffect(element) {
    element.style.boxShadow = '';
    element.style.transform = '';
    element.style.borderColor = '';
    
    const beam = element.parentElement.querySelector('.focus-beam');
    if (beam) {
        beam.remove();
    }
}

/**
 * تأثير الكتابة المتقدم
 */
function addAdvancedTypingEffect(element) {
    element.style.background = 'linear-gradient(135deg, #ffffff 0%, #f0f8ff 100%)';
    element.style.borderColor = '#667eea';
    
    // تأثير النبضة
    const pulse = document.createElement('div');
    pulse.className = 'typing-pulse';
    pulse.style.cssText = `
        position: absolute;
        top: 50%;
        right: 10px;
        width: 8px;
        height: 8px;
        background: #667eea;
        border-radius: 50%;
        transform: translateY(-50%);
        animation: pulse 1s ease-in-out infinite;
        pointer-events: none;
        z-index: 10;
    `;
    
    element.parentElement.style.position = 'relative';
    element.parentElement.appendChild(pulse);
    
    clearTimeout(animationTimeout);
    animationTimeout = setTimeout(() => {
        element.style.background = '';
        element.style.borderColor = '';
        if (pulse.parentNode) pulse.remove();
    }, 1000);
}

/**
 * تأثير اختيار القائمة المنسدلة
 */
function addSelectionAnimation(select) {
    const sparkle = document.createElement('div');
    sparkle.className = 'selection-sparkle';
    sparkle.style.cssText = `
        position: absolute;
        top: 50%;
        left: 50%;
        width: 20px;
        height: 20px;
        background: radial-gradient(circle, #667eea, transparent);
        border-radius: 50%;
        transform: translate(-50%, -50%) scale(0);
        animation: sparkle 0.6s ease-out;
        pointer-events: none;
        z-index: 10;
    `;
    
    select.parentElement.style.position = 'relative';
    select.parentElement.appendChild(sparkle);
    
    setTimeout(() => {
        if (sparkle.parentNode) sparkle.remove();
    }, 600);
}

/**
 * تهيئة اختصارات لوحة المفاتيح
 */
function initializeKeyboardShortcuts() {
    document.addEventListener('keydown', function(e) {
        // Ctrl+S للحفظ
        if (e.ctrlKey && e.key === 's') {
            e.preventDefault();
            saveForm();
        }
        
        // Ctrl+Enter للحفظ والإرسال
        if (e.ctrlKey && e.key === 'Enter') {
            e.preventDefault();
            saveAndSendForm();
        }
        
        // Ctrl+N لإضافة عنصر جديد
        if (e.ctrlKey && e.key === 'n') {
            e.preventDefault();
            addInvoiceItem();
        }
        
        // Escape لإلغاء المودال
        if (e.key === 'Escape') {
            closeModals();
        }
    });
}

/**
 * معالجة اختصارات لوحة المفاتيح في الحقول
 */
function handleKeyboardShortcuts(e, element) {
    // Tab للانتقال المحسن
    if (e.key === 'Tab') {
        addTabTransition(element);
    }
    
    // Enter في الحقول الرقمية للحساب
    if (e.key === 'Enter' && element.type === 'number') {
        e.preventDefault();
        calculateTotals();
        moveFocusToNext(element);
    }
}

/**
 * تأثير انتقال Tab
 */
function addTabTransition(element) {
    element.style.transition = 'all 0.3s ease';
    element.style.borderColor = '#10b981';
    
    setTimeout(() => {
        element.style.borderColor = '';
    }, 300);
}

/**
 * الانتقال للحقل التالي
 */
function moveFocusToNext(currentElement) {
    const formElements = Array.from(document.querySelectorAll('input, select, textarea'));
    const currentIndex = formElements.indexOf(currentElement);
    const nextElement = formElements[currentIndex + 1];
    
    if (nextElement) {
        nextElement.focus();
        addAdvancedGlowEffect(nextElement);
    }
}

/**
 * تهيئة التلميحات المتقدمة
 */
function initializeAdvancedTooltips() {
    const tooltipElements = document.querySelectorAll('[data-tooltip]');
    tooltipElements.forEach(element => {
        element.addEventListener('mouseenter', showAdvancedTooltip);
        element.addEventListener('mouseleave', hideAdvancedTooltip);
    });
}

/**
 * عرض التلميح المتقدم
 */
function showAdvancedTooltip(e) {
    const tooltip = document.createElement('div');
    tooltip.className = 'advanced-tooltip';
    tooltip.textContent = e.target.getAttribute('data-tooltip');
    tooltip.style.cssText = `
        position: absolute;
        background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
        color: white;
        padding: 12px 16px;
        border-radius: 12px;
        font-size: 0.875rem;
        box-shadow: 0 10px 25px rgba(0,0,0,0.2);
        z-index: 10000;
        opacity: 0;
        transform: translateY(10px);
        transition: all 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94);
        pointer-events: none;
        white-space: nowrap;
        backdrop-filter: blur(10px);
    `;
    
    document.body.appendChild(tooltip);
    
    const rect = e.target.getBoundingClientRect();
    tooltip.style.left = rect.left + (rect.width / 2) - (tooltip.offsetWidth / 2) + 'px';
    tooltip.style.top = rect.top - tooltip.offsetHeight - 10 + 'px';
    
    setTimeout(() => {
        tooltip.style.opacity = '1';
        tooltip.style.transform = 'translateY(0)';
    }, 50);
    
    e.target._tooltip = tooltip;
}

/**
 * إخفاء التلميح المتقدم
 */
function hideAdvancedTooltip(e) {
    if (e.target._tooltip) {
        const tooltip = e.target._tooltip;
        tooltip.style.opacity = '0';
        tooltip.style.transform = 'translateY(10px)';
        setTimeout(() => {
            if (tooltip.parentNode) tooltip.remove();
        }, 300);
        delete e.target._tooltip;
    }
}

/**
 * تهيئة الحفظ التلقائي
 */
function initializeAutoSave() {
    let autoSaveTimeout;
    
    function scheduleAutoSave() {
        clearTimeout(autoSaveTimeout);
        autoSaveTimeout = setTimeout(() => {
            autoSaveForm();
        }, 30000); // حفظ كل 30 ثانية
    }
    
    // ربط بأحداث التغيير
    document.addEventListener('input', scheduleAutoSave);
    document.addEventListener('change', scheduleAutoSave);
    
    window.scheduleAutoSave = scheduleAutoSave;
}

/**
 * حفظ تلقائي للنموذج
 */
function autoSaveForm() {
    const formData = new FormData(document.getElementById('invoice-form'));
    const data = Object.fromEntries(formData.entries());
    
    // حفظ في localStorage
    localStorage.setItem('invoice_draft', JSON.stringify({
        data: data,
        timestamp: new Date().toISOString(),
        items: getInvoiceItemsData()
    }));
    
    showQuietNotification('تم الحفظ التلقائي', 'success');
}

/**
 * إعداد التحقق المتقدم من النموذج
 */
function setupAdvancedFormValidation() {
    const form = document.getElementById('invoice-form');
    if (form) {
        form.addEventListener('submit', function(e) {
            if (!validateFormAdvanced()) {
                e.preventDefault();
                showValidationSummary();
            } else {
                showSubmissionAnimation();
            }
        });
    }
    
    // تحقق فوري من الحقول
    const inputs = document.querySelectorAll('.form-control-professional');
    inputs.forEach(input => {
        input.addEventListener('blur', () => validateFieldInstantly(input));
    });
}

/**
 * التحقق المتقدم من النموذج
 */
function validateFormAdvanced() {
    validationErrors = [];
    const requiredFields = document.querySelectorAll('[required]');
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            validationErrors.push({
                field: field,
                message: `حقل "${getFieldLabel(field)}" مطلوب`,
                type: 'required'
            });
        }
    });
    
    // التحقق من وجود عناصر
    const items = document.querySelectorAll('#invoice-items .item-row');
    if (items.length === 0) {
        validationErrors.push({
            field: null,
            message: 'يجب إضافة عنصر واحد على الأقل للفاتورة',
            type: 'items'
        });
    }
    
    // التحقق من صحة المبالغ
    validateAmounts();
    
    // التحقق من صحة التواريخ
    validateDates();
    
    return validationErrors.length === 0;
}

/**
 * التحقق الفوري من الحقل
 */
function validateFieldInstantly(field) {
    clearFieldError(field);
    
    if (field.hasAttribute('required') && !field.value.trim()) {
        showFieldError(field, 'هذا الحقل مطلوب');
        return false;
    }
    
    // تحقق مخصص حسب نوع الحقل
    if (field.type === 'email' && field.value && !isValidEmail(field.value)) {
        showFieldError(field, 'عنوان البريد الإلكتروني غير صحيح');
        return false;
    }
    
    if (field.type === 'number' && field.value && parseFloat(field.value) < 0) {
        showFieldError(field, 'القيمة يجب أن تكون أكبر من أو تساوي صفر');
        return false;
    }
    
    showFieldSuccess(field);
    return true;
}

/**
 * عرض خطأ الحقل
 */
function showFieldError(field, message) {
    field.classList.add('is-invalid');
    field.style.borderColor = '#ef4444';
    field.style.boxShadow = '0 0 0 3px rgba(239, 68, 68, 0.1)';
    
    // إضافة رسالة الخطأ
    let errorDiv = field.parentElement.querySelector('.field-error');
    if (!errorDiv) {
        errorDiv = document.createElement('div');
        errorDiv.className = 'field-error';
        errorDiv.style.cssText = `
            color: #ef4444;
            font-size: 0.875rem;
            margin-top: 0.25rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            animation: errorSlideIn 0.3s ease;
        `;
        field.parentElement.appendChild(errorDiv);
    }
    
    errorDiv.innerHTML = `<i class="fas fa-exclamation-triangle"></i>${message}`;
}

/**
 * إزالة خطأ الحقل
 */
function clearFieldError(field) {
    field.classList.remove('is-invalid');
    field.style.borderColor = '';
    field.style.boxShadow = '';
    
    const errorDiv = field.parentElement.querySelector('.field-error');
    if (errorDiv) {
        errorDiv.remove();
    }
}

/**
 * عرض نجاح الحقل
 */
function showFieldSuccess(field) {
    if (field.value.trim()) {
        field.classList.add('is-valid');
        field.style.borderColor = '#10b981';
        
        // إضافة علامة النجاح
        let successIcon = field.parentElement.querySelector('.field-success-icon');
        if (!successIcon) {
            successIcon = document.createElement('div');
            successIcon.className = 'field-success-icon';
            successIcon.style.cssText = `
                position: absolute;
                left: 0.75rem;
                top: 50%;
                transform: translateY(-50%);
                color: #10b981;
                font-size: 1rem;
                animation: successPop 0.3s ease;
                pointer-events: none;
                z-index: 10;
            `;
            successIcon.innerHTML = '<i class="fas fa-check"></i>';
            field.parentElement.style.position = 'relative';
            field.parentElement.appendChild(successIcon);
        }
        
        setTimeout(() => {
            field.classList.remove('is-valid');
            field.style.borderColor = '';
            if (successIcon.parentNode) successIcon.remove();
        }, 2000);
    }
}

/**
 * التحقق من صحة المبالغ
 */
function validateAmounts() {
    const totalAmountText = document.getElementById('total-display').textContent;
    const totalAmount = parseFloat(totalAmountText.replace(' ر.س', '')) || 0;
    const paidAmount = parseFloat(document.querySelector('input[name="paid_amount"]').value) || 0;
    
    if (paidAmount > totalAmount) {
        validationErrors.push({
            field: document.querySelector('input[name="paid_amount"]'),
            message: 'المبلغ المدفوع لا يمكن أن يكون أكبر من المبلغ الإجمالي',
            type: 'amount'
        });
    }
}

/**
 * التحقق من صحة التواريخ
 */
function validateDates() {
    const issueDate = document.querySelector('input[name="issue_date"]').value;
    const dueDate = document.querySelector('input[name="due_date"]').value;
    
    if (issueDate && dueDate && new Date(dueDate) < new Date(issueDate)) {
        validationErrors.push({
            field: document.querySelector('input[name="due_date"]'),
            message: 'تاريخ الاستحقاق يجب أن يكون بعد تاريخ الإصدار',
            type: 'date'
        });
    }
}

/**
 * عرض ملخص أخطاء التحقق
 */
function showValidationSummary() {
    const errorSummary = document.createElement('div');
    errorSummary.className = 'validation-summary';
    errorSummary.style.cssText = `
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        background: white;
        border-radius: 1rem;
        box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        padding: 2rem;
        z-index: 10000;
        max-width: 500px;
        width: 90%;
        max-height: 80vh;
        overflow-y: auto;
        animation: modalAppear 0.3s ease;
    `;
    
    let errorsHtml = `
        <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1.5rem;">
            <div style="width: 50px; height: 50px; background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; color: white; font-size: 1.5rem;">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <div>
                <h3 style="margin: 0; color: #1e293b; font-size: 1.25rem;">يرجى تصحيح الأخطاء التالية:</h3>
                <p style="margin: 0.25rem 0 0 0; color: #64748b;">تم العثور على ${validationErrors.length} خطأ في النموذج</p>
            </div>
        </div>
        <ul style="list-style: none; padding: 0; margin: 0;">
    `;
    
    validationErrors.forEach((error, index) => {
        errorsHtml += `
            <li style="padding: 0.75rem; margin-bottom: 0.5rem; background: #fef2f2; border-radius: 0.5rem; border-right: 4px solid #ef4444; cursor: pointer;" onclick="focusErrorField(${index})">
                <div style="display: flex; align-items: center; gap: 0.5rem;">
                    <i class="fas fa-times-circle" style="color: #ef4444;"></i>
                    <span style="color: #7f1d1d; font-weight: 500;">${error.message}</span>
                </div>
            </li>
        `;
    });
    
    errorsHtml += `
        </ul>
        <div style="margin-top: 1.5rem; display: flex; gap: 0.75rem;">
            <button onclick="closeValidationSummary()" style="flex: 1; background: #ef4444; color: white; border: none; padding: 0.75rem 1.5rem; border-radius: 0.5rem; font-weight: 600; cursor: pointer; transition: all 0.3s ease;">
                إغلاق
            </button>
        </div>
    `;
    
    errorSummary.innerHTML = errorsHtml;
    
    // إضافة overlay
    const overlay = document.createElement('div');
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.5);
        z-index: 9999;
        animation: overlayAppear 0.3s ease;
    `;
    overlay.onclick = closeValidationSummary;
    
    document.body.appendChild(overlay);
    document.body.appendChild(errorSummary);
    
    window.currentValidationSummary = errorSummary;
    window.currentValidationOverlay = overlay;
    window.validationErrors = validationErrors;
}

/**
 * إغلاق ملخص التحقق
 */
function closeValidationSummary() {
    if (window.currentValidationSummary) {
        window.currentValidationSummary.remove();
        window.currentValidationOverlay.remove();
        delete window.currentValidationSummary;
        delete window.currentValidationOverlay;
    }
}

/**
 * التركيز على حقل الخطأ
 */
function focusErrorField(errorIndex) {
    const error = window.validationErrors[errorIndex];
    if (error.field) {
        error.field.focus();
        error.field.scrollIntoView({ behavior: 'smooth', block: 'center' });
        addAdvancedGlowEffect(error.field);
    }
    closeValidationSummary();
}

/**
 * تحديث مؤشر تقدم النموذج
 */
function updateFormProgress() {
    const progressSteps = document.querySelectorAll('.step-indicator');
    const formData = new FormData(document.getElementById('invoice-form'));
    
    let completedSteps = 0;
    
    // الخطوة 1: معلومات أساسية
    if (formData.get('type') && formData.get('issue_date')) {
        completedSteps = Math.max(completedSteps, 1);
    }
    
    // الخطوة 2: العناصر
    if (document.querySelectorAll('#invoice-items .item-row').length > 0) {
        completedSteps = Math.max(completedSteps, 2);
    }
    
    // الخطوة 3: السداد
    if (formData.get('payment_method') || formData.get('paid_amount')) {
        completedSteps = Math.max(completedSteps, 3);
    }
    
    // الخطوة 4: الإشعارات
    if (formData.get('send_creditor_notification') || formData.get('send_debtor_notification')) {
        completedSteps = Math.max(completedSteps, 4);
    }
    
    // تحديث المؤشرات
    progressSteps.forEach((step, index) => {
        const stepNumber = parseInt(step.querySelector('.step-number').textContent);
        if (stepNumber <= completedSteps) {
            step.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
            step.style.boxShadow = '0 4px 15px rgba(16, 185, 129, 0.3)';
            step.style.transform = 'scale(1.1)';
        } else {
            step.style.background = 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)';
            step.style.boxShadow = '0 4px 15px rgba(102, 126, 234, 0.3)';
            step.style.transform = 'scale(1)';
        }
    });
}

/**
 * تشغيل صوت التركيز
 */
function playFocusSound() {
    // يمكن إضافة صوت لطيف هنا في المستقبل
    // const audio = new Audio('/sounds/focus.mp3');
    // audio.volume = 0.1;
    // audio.play().catch(() => {});
}

/**
 * عرض إشعار هادئ
 */
function showQuietNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = 'quiet-notification';
    notification.style.cssText = `
        position: fixed;
        bottom: 20px;
        right: 20px;
        background: ${getNotificationColor(type)};
        color: white;
        padding: 0.75rem 1rem;
        border-radius: 0.5rem;
        font-size: 0.875rem;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        z-index: 9999;
        opacity: 0;
        transform: translateY(20px);
        transition: all 0.3s ease;
        max-width: 300px;
    `;
    
    notification.textContent = message;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.opacity = '1';
        notification.style.transform = 'translateY(0)';
    }, 50);
    
    setTimeout(() => {
        notification.style.opacity = '0';
        notification.style.transform = 'translateY(20px)';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

/**
 * الحصول على لون الإشعار
 */
function getNotificationColor(type) {
    const colors = {
        'success': 'linear-gradient(135deg, #10b981 0%, #059669 100%)',
        'error': 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)',
        'warning': 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)',
        'info': 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)'
    };
    return colors[type] || colors.info;
}

/**
 * دوال مساعدة
 */
function getFieldLabel(field) {
    const label = field.parentElement.querySelector('label');
    return label ? label.textContent.replace('*', '').trim() : field.name;
}

function isValidEmail(email) {
    return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
}

function getInvoiceItemsData() {
    const items = [];
    document.querySelectorAll('#invoice-items .item-row').forEach(row => {
        const productId = row.querySelector('input[name*="[product_id]"]').value;
        const quantity = row.querySelector('input[name*="[quantity]"]').value;
        const unitPrice = row.querySelector('input[name*="[unit_price]"]').value;
        const discount = row.querySelector('input[name*="[discount_amount]"]').value;
        
        items.push({
            product_id: productId,
            quantity: quantity,
            unit_price: unitPrice,
            discount_amount: discount
        });
    });
    return items;
}

function saveForm() {
    document.getElementById('invoice-form').submit();
}

function saveAndSendForm() {
    const form = document.getElementById('invoice-form');
    const actionInput = document.createElement('input');
    actionInput.type = 'hidden';
    actionInput.name = 'action';
    actionInput.value = 'save_and_send';
    form.appendChild(actionInput);
    form.submit();
}

function closeModals() {
    const modals = document.querySelectorAll('.modal.show');
    modals.forEach(modal => {
        const bsModal = bootstrap.Modal.getInstance(modal);
        if (bsModal) bsModal.hide();
    });
}

// إضافة CSS للرسوم المتحركة الجديدة
const enhancedAnimationStyles = document.createElement('style');
enhancedAnimationStyles.textContent = `
    @keyframes beamMove {
        0% { transform: translateX(-100%); }
        100% { transform: translateX(100%); }
    }
    
    @keyframes pulse {
        0%, 100% { opacity: 1; transform: translateY(-50%) scale(1); }
        50% { opacity: 0.5; transform: translateY(-50%) scale(1.2); }
    }
    
    @keyframes sparkle {
        0% { transform: translate(-50%, -50%) scale(0) rotate(0deg); opacity: 1; }
        50% { transform: translate(-50%, -50%) scale(1) rotate(180deg); opacity: 0.8; }
        100% { transform: translate(-50%, -50%) scale(0) rotate(360deg); opacity: 0; }
    }
    
    @keyframes errorSlideIn {
        from { opacity: 0; transform: translateY(-10px); }
        to { opacity: 1; transform: translateY(0); }
    }
    
    @keyframes successPop {
        0% { transform: translateY(-50%) scale(0); }
        50% { transform: translateY(-50%) scale(1.2); }
        100% { transform: translateY(-50%) scale(1); }
    }
    
    @keyframes modalAppear {
        from { opacity: 0; transform: translate(-50%, -50%) scale(0.9); }
        to { opacity: 1; transform: translate(-50%, -50%) scale(1); }
    }
    
    @keyframes overlayAppear {
        from { opacity: 0; }
        to { opacity: 1; }
    }
    
    .field-focused {
        position: relative;
        z-index: 10;
    }
    
    .form-control-professional.is-valid {
        border-color: #10b981;
        box-shadow: 0 0 0 3px rgba(16, 185, 129, 0.1);
    }
    
    .professional-card {
        transform: translateY(0);
        transition: all 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94);
    }
    
    .professional-card:hover {
        transform: translateY(-4px);
    }
`;

document.head.appendChild(enhancedAnimationStyles);

// تحديث عداد العناصر عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    updateItemsCount();
});

// إعداد أحداث تحديث التقدم
document.addEventListener('change', updateFormProgress);
document.addEventListener('input', updateFormProgress);