// نظام المبيعات المتوافق مع ZATCA - JavaScript الرئيسي

// إعدادات عامة
const CONFIG = {
    currency: 'ر.س',
    locale: 'ar-SA',
    dateFormat: 'YYYY-MM-DD',
    taxRate: 15, // معدل الضريبة الافتراضي
    ajax: {
        timeout: 30000,
        retries: 3
    }
};

// وظائف المساعدة
const Utils = {
    // تنسيق الأرقام
    formatNumber: function(number, decimals = 2) {
        return new Intl.NumberFormat(CONFIG.locale, {
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals
        }).format(number);
    },

    // تنسيق العملة
    formatCurrency: function(amount, decimals = 2) {
        return this.formatNumber(amount, decimals) + ' ' + CONFIG.currency;
    },

    // تنسيق التاريخ
    formatDate: function(date) {
        return new Date(date).toLocaleDateString(CONFIG.locale);
    },

    // إخفاء النص الحساس
    maskText: function(text, visibleChars = 4) {
        if (!text || text.length <= visibleChars * 2) return text;
        const start = text.substring(0, visibleChars);
        const end = text.substring(text.length - visibleChars);
        const middle = '*'.repeat(text.length - visibleChars * 2);
        return start + middle + end;
    },

    // التحقق من صحة الرقم الضريبي السعودي
    validateVatNumber: function(vatNumber) {
        if (!vatNumber) return false;
        
        // يجب أن يكون 15 رقم ويبدأ بـ 3
        const regex = /^3[0-9]{14}$/;
        return regex.test(vatNumber);
    },

    // التحقق من صحة الهوية الوطنية
    validateNationalId: function(nationalId) {
        if (!nationalId || nationalId.length !== 10) return false;
        
        // خوارزمية التحقق من صحة الهوية الوطنية السعودية
        let sum = 0;
        for (let i = 0; i < 9; i++) {
            const digit = parseInt(nationalId[i]);
            if (i % 2 === 0) {
                sum += digit;
            } else {
                const doubled = digit * 2;
                sum += doubled > 9 ? doubled - 9 : doubled;
            }
        }
        
        const checkDigit = (10 - (sum % 10)) % 10;
        return checkDigit === parseInt(nationalId[9]);
    },

    // عرض رسالة تنبيه
    showAlert: function(message, type = 'info', duration = 5000) {
        const alertHtml = `
            <div class="alert alert-${type} alert-dismissible fade show position-fixed" 
                 style="top: 20px; right: 20px; z-index: 9999; min-width: 300px;" role="alert">
                <i class="fas fa-${this.getAlertIcon(type)} me-2"></i>
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        `;
        
        document.body.insertAdjacentHTML('beforeend', alertHtml);
        
        // إزالة التنبيه تلقائياً
        setTimeout(() => {
            const alert = document.querySelector('.alert:last-of-type');
            if (alert) {
                alert.remove();
            }
        }, duration);
    },

    // الحصول على أيقونة التنبيه
    getAlertIcon: function(type) {
        const icons = {
            success: 'check-circle',
            danger: 'exclamation-triangle',
            warning: 'exclamation-triangle',
            info: 'info-circle'
        };
        return icons[type] || 'info-circle';
    },

    // تحديث الـ URL بدون إعادة تحميل الصفحة
    updateUrl: function(params) {
        const url = new URL(window.location);
        Object.keys(params).forEach(key => {
            if (params[key]) {
                url.searchParams.set(key, params[key]);
            } else {
                url.searchParams.delete(key);
            }
        });
        window.history.pushState({}, '', url);
    }
};

// وظائف AJAX
const Ajax = {
    // إرسال طلب AJAX
    request: function(url, options = {}) {
        const defaults = {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content')
            },
            timeout: CONFIG.ajax.timeout
        };

        const config = { ...defaults, ...options };

        return fetch(url, config)
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                return response.json();
            })
            .catch(error => {
                console.error('AJAX Error:', error);
                throw error;
            });
    },

    // GET request
    get: function(url, params = {}) {
        const queryString = new URLSearchParams(params).toString();
        const fullUrl = queryString ? `${url}?${queryString}` : url;
        return this.request(fullUrl);
    },

    // POST request
    post: function(url, data = {}) {
        return this.request(url, {
            method: 'POST',
            body: JSON.stringify(data)
        });
    },

    // PUT request
    put: function(url, data = {}) {
        return this.request(url, {
            method: 'PUT',
            body: JSON.stringify(data)
        });
    },

    // DELETE request
    delete: function(url) {
        return this.request(url, {
            method: 'DELETE'
        });
    }
};

// إدارة النماذج
const Forms = {
    // التحقق من صحة النموذج
    validate: function(form) {
        const inputs = form.querySelectorAll('input[required], select[required], textarea[required]');
        let isValid = true;
        
        inputs.forEach(input => {
            if (!input.value.trim()) {
                this.showFieldError(input, 'هذا الحقل مطلوب');
                isValid = false;
            } else {
                this.clearFieldError(input);
            }
        });

        return isValid;
    },

    // عرض خطأ في الحقل
    showFieldError: function(field, message) {
        field.classList.add('is-invalid');
        
        let feedback = field.parentNode.querySelector('.invalid-feedback');
        if (!feedback) {
            feedback = document.createElement('div');
            feedback.className = 'invalid-feedback';
            field.parentNode.appendChild(feedback);
        }
        feedback.textContent = message;
    },

    // إزالة خطأ الحقل
    clearFieldError: function(field) {
        field.classList.remove('is-invalid');
        const feedback = field.parentNode.querySelector('.invalid-feedback');
        if (feedback) {
            feedback.remove();
        }
    },

    // إرسال النموذج بـ AJAX
    submit: function(form, onSuccess, onError) {
        const formData = new FormData(form);
        const data = Object.fromEntries(formData);
        
        const submitButton = form.querySelector('button[type="submit"]');
        const originalText = submitButton.innerHTML;
        
        submitButton.disabled = true;
        submitButton.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>جاري الحفظ...';

        Ajax.post(form.action, data)
            .then(response => {
                if (onSuccess) onSuccess(response);
                Utils.showAlert('تم الحفظ بنجاح', 'success');
            })
            .catch(error => {
                if (onError) onError(error);
                Utils.showAlert('حدث خطأ أثناء الحفظ', 'danger');
            })
            .finally(() => {
                submitButton.disabled = false;
                submitButton.innerHTML = originalText;
            });
    }
};

// إدارة الجداول
const Tables = {
    // تطبيق البحث على الجدول
    setupSearch: function(searchInput, table) {
        searchInput.addEventListener('input', function() {
            const query = this.value.toLowerCase();
            const rows = table.querySelectorAll('tbody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(query) ? '' : 'none';
            });
        });
    },

    // تطبيق الفلترة
    setupFilter: function(filterSelect, table, columnIndex) {
        filterSelect.addEventListener('change', function() {
            const filterValue = this.value.toLowerCase();
            const rows = table.querySelectorAll('tbody tr');
            
            rows.forEach(row => {
                const cell = row.children[columnIndex];
                if (cell) {
                    const cellText = cell.textContent.toLowerCase();
                    row.style.display = !filterValue || cellText.includes(filterValue) ? '' : 'none';
                }
            });
        });
    },

    // ترتيب الجدول
    setupSort: function(table) {
        const headers = table.querySelectorAll('thead th[data-sort]');
        
        headers.forEach(header => {
            header.style.cursor = 'pointer';
            header.addEventListener('click', () => {
                const column = header.dataset.sort;
                const type = header.dataset.type || 'text';
                this.sortTable(table, column, type);
            });
        });
    },

    // تنفيذ الترتيب
    sortTable: function(table, column, type) {
        const tbody = table.querySelector('tbody');
        const rows = Array.from(tbody.querySelectorAll('tr'));
        
        rows.sort((a, b) => {
            const aVal = a.children[column]?.textContent.trim() || '';
            const bVal = b.children[column]?.textContent.trim() || '';
            
            if (type === 'number') {
                return parseFloat(aVal) - parseFloat(bVal);
            } else if (type === 'date') {
                return new Date(aVal) - new Date(bVal);
            } else {
                return aVal.localeCompare(bVal, CONFIG.locale);
            }
        });
        
        rows.forEach(row => tbody.appendChild(row));
    }
};

// إدارة الفواتير
const Invoices = {
    // حساب إجمالي السطر
    calculateLineTotal: function(quantity, price, discount = 0, taxRate = CONFIG.taxRate) {
        const subtotal = (quantity * price) - discount;
        const tax = subtotal * (taxRate / 100);
        return {
            subtotal: subtotal,
            tax: tax,
            total: subtotal + tax
        };
    },

    // حساب إجمالي الفاتورة
    calculateInvoiceTotal: function(items, globalDiscount = 0) {
        let subtotal = 0;
        let totalTax = 0;
        
        items.forEach(item => {
            const lineCalc = this.calculateLineTotal(
                item.quantity, 
                item.price, 
                item.discount, 
                item.taxRate
            );
            subtotal += lineCalc.subtotal;
            totalTax += lineCalc.tax;
        });
        
        const finalTotal = subtotal + totalTax - globalDiscount;
        
        return {
            subtotal: subtotal,
            tax: totalTax,
            discount: globalDiscount,
            total: finalTotal
        };
    },

    // إضافة عنصر جديد للفاتورة
    addInvoiceItem: function(container, item) {
        const index = container.children.length;
        const template = this.getItemTemplate(index, item);
        container.insertAdjacentHTML('beforeend', template);
        
        // ربط الأحداث
        this.bindItemEvents(container.lastElementChild, index);
    },

    // قالب عنصر الفاتورة
    getItemTemplate: function(index, item = {}) {
        return `
            <tr id="item-${index}" data-index="${index}">
                <td>
                    <input type="hidden" name="items[${index}][product_id]" value="${item.product_id || ''}">
                    <strong class="item-name">${item.name || ''}</strong>
                    <br><small class="text-muted item-description">${item.description || ''}</small>
                    <br><small class="text-info">رمز: <span class="item-sku">${item.sku || ''}</span></small>
                </td>
                <td>
                    <input type="number" name="items[${index}][quantity]" class="form-control item-quantity" 
                           value="${item.quantity || 1}" step="0.001" min="0.001" required>
                </td>
                <td>
                    <input type="number" name="items[${index}][unit_price]" class="form-control item-price" 
                           value="${item.price || 0}" step="0.01" min="0" required>
                </td>
                <td>
                    <input type="number" name="items[${index}][discount_amount]" class="form-control item-discount" 
                           value="${item.discount || 0}" step="0.01" min="0">
                </td>
                <td class="item-subtotal">${Utils.formatCurrency(item.subtotal || 0)}</td>
                <td class="item-tax">${Utils.formatCurrency(item.tax || 0)}</td>
                <td class="item-total">${Utils.formatCurrency(item.total || 0)}</td>
                <td>
                    <button type="button" class="btn btn-danger btn-sm remove-item">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `;
    },

    // ربط أحداث عنصر الفاتورة
    bindItemEvents: function(row, index) {
        const quantityInput = row.querySelector('.item-quantity');
        const priceInput = row.querySelector('.item-price');
        const discountInput = row.querySelector('.item-discount');
        const removeButton = row.querySelector('.remove-item');

        [quantityInput, priceInput, discountInput].forEach(input => {
            input.addEventListener('input', () => this.updateItemCalculations(row));
        });

        removeButton.addEventListener('click', () => {
            row.remove();
            this.updateInvoiceTotal();
        });

        // حساب أولي
        this.updateItemCalculations(row);
    },

    // تحديث حسابات العنصر
    updateItemCalculations: function(row) {
        const quantity = parseFloat(row.querySelector('.item-quantity').value) || 0;
        const price = parseFloat(row.querySelector('.item-price').value) || 0;
        const discount = parseFloat(row.querySelector('.item-discount').value) || 0;
        
        const calculations = this.calculateLineTotal(quantity, price, discount);
        
        row.querySelector('.item-subtotal').textContent = Utils.formatCurrency(calculations.subtotal);
        row.querySelector('.item-tax').textContent = Utils.formatCurrency(calculations.tax);
        row.querySelector('.item-total').textContent = Utils.formatCurrency(calculations.total);
        
        this.updateInvoiceTotal();
    },

    // تحديث إجمالي الفاتورة
    updateInvoiceTotal: function() {
        const items = Array.from(document.querySelectorAll('#invoice-items tr')).map(row => ({
            quantity: parseFloat(row.querySelector('.item-quantity')?.value) || 0,
            price: parseFloat(row.querySelector('.item-price')?.value) || 0,
            discount: parseFloat(row.querySelector('.item-discount')?.value) || 0,
            taxRate: CONFIG.taxRate
        }));

        const globalDiscount = parseFloat(document.querySelector('input[name="discount_amount"]')?.value) || 0;
        const totals = this.calculateInvoiceTotal(items, globalDiscount);

        // تحديث عناصر العرض
        const subtotalElement = document.getElementById('subtotal-display');
        const taxElement = document.getElementById('tax-total-display');
        const totalElement = document.getElementById('total-display');

        if (subtotalElement) subtotalElement.textContent = Utils.formatCurrency(totals.subtotal);
        if (taxElement) taxElement.textContent = Utils.formatCurrency(totals.tax);
        if (totalElement) totalElement.textContent = Utils.formatCurrency(totals.total);
    }
};

// إدارة المنتجات
const Products = {
    // حساب السعر مع الضريبة
    calculatePriceWithTax: function(price, taxRate, isTaxable = true) {
        if (!isTaxable) return price;
        return price * (1 + (taxRate / 100));
    },

    // تحديث حسابات المنتج
    updateProductCalculations: function() {
        const priceInput = document.querySelector('input[name="price"]');
        const taxRateInput = document.querySelector('input[name="tax_rate"]');
        const isTaxableInput = document.querySelector('input[name="is_taxable"]');
        const priceWithTaxDisplay = document.getElementById('price-with-tax-display');

        if (priceInput && taxRateInput && priceWithTaxDisplay) {
            const price = parseFloat(priceInput.value) || 0;
            const taxRate = parseFloat(taxRateInput.value) || 0;
            const isTaxable = isTaxableInput ? isTaxableInput.checked : true;

            const priceWithTax = this.calculatePriceWithTax(price, taxRate, isTaxable);
            priceWithTaxDisplay.value = Utils.formatNumber(priceWithTax);
        }
    }
};

// تهيئة النظام عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 نظام المبيعات ZATCA محمل وجاهز');

    // إعداد التنبيهات التلقائية للنماذج
    document.querySelectorAll('form').forEach(form => {
        form.addEventListener('submit', function(e) {
            if (!Forms.validate(this)) {
                e.preventDefault();
                Utils.showAlert('يرجى تصحيح الأخطاء في النموذج', 'warning');
            }
        });
    });

    // إعداد البحث في الجداول
    document.querySelectorAll('[data-search-table]').forEach(searchInput => {
        const tableId = searchInput.dataset.searchTable;
        const table = document.getElementById(tableId);
        if (table) {
            Tables.setupSearch(searchInput, table);
        }
    });

    // إعداد الترتيب في الجداول
    document.querySelectorAll('table[data-sortable]').forEach(table => {
        Tables.setupSort(table);
    });

    // إعداد التحقق من الأرقام الضريبية
    document.querySelectorAll('input[name="vat_number"]').forEach(input => {
        input.addEventListener('blur', function() {
            if (this.value && !Utils.validateVatNumber(this.value)) {
                Forms.showFieldError(this, 'رقم ضريبي غير صحيح (يجب أن يكون 15 رقم ويبدأ بـ 3)');
            } else {
                Forms.clearFieldError(this);
            }
        });
    });

    // إعداد التحقق من الهوية الوطنية
    document.querySelectorAll('input[name="national_id"]').forEach(input => {
        input.addEventListener('blur', function() {
            if (this.value && !Utils.validateNationalId(this.value)) {
                Forms.showFieldError(this, 'رقم هوية وطنية غير صحيح');
            } else {
                Forms.clearFieldError(this);
            }
        });
    });

    // تهيئة حسابات المنتجات
    if (document.querySelector('input[name="price"]')) {
        Products.updateProductCalculations();
        
        ['input[name="price"]', 'input[name="tax_rate"]', 'input[name="is_taxable"]'].forEach(selector => {
            const element = document.querySelector(selector);
            if (element) {
                element.addEventListener('input', Products.updateProductCalculations);
                element.addEventListener('change', Products.updateProductCalculations);
            }
        });
    }

    // تهيئة حسابات الفواتير
    if (document.getElementById('invoice-items')) {
        // ربط أحداث العناصر الموجودة
        document.querySelectorAll('#invoice-items tr').forEach((row, index) => {
            Invoices.bindItemEvents(row, index);
        });

        // ربط حدث الخصم العام
        const discountInput = document.querySelector('input[name="discount_amount"]');
        if (discountInput) {
            discountInput.addEventListener('input', Invoices.updateInvoiceTotal);
        }
    }

    console.log('✅ تم تهيئة جميع المكونات بنجاح');
});

// تصدير الوظائف للاستخدام العام
window.ZatcaApp = {
    Utils,
    Ajax,
    Forms,
    Tables,
    Invoices,
    Products,
    CONFIG
};